/* cmstat.c: print stuff about /dev/cm devices */
#include <stdio.h>
#include <sys/types.h>
#include <nlist.h>

/* include stuff from en.c */
#include <sys/enio.h>
#include <sys/ethernet.h>
#include <sys/stream.h>

/* configuration */

#define	NTYPES		8	/* max number of ethernet packet types */
#define	NRRB		64	/* number of recieve buffers */
#define	LOGNRRB		6	/* log of NRRB*/
#define	NTRB		32	/* number of transmit buffers */
#define	LOGNTRB		5	/* log of NTRB */
#define	ENPRI		28	/* sleeping like a tty */
#define	SPLIMP		6	/* interrupt levell */
#define	LANCEAP		((u_short *)0xfffe1802)
#define	LANCEDP		((u_short *)0xfffe1800)
#define	NRBLOCKS		(NTRB+NRRB+100)


#define	RSUCC(x)	(((x)+1)%NRRB)
#define	TSUCC(x)	(((x)+1)%NTRB)

/* macro to map an address into the no cache segment */

#define	NCADDR(t, x)	((t *)(0x81000000|(u_long)(x)))

/*   mode bits in the lance initialization block */

#define	PROM	0x8000
#define	INTL	0x40
#define	DRTY	0x20
#define	COLL	0x10
#define	DTCR	0x8
#define	LOOP	0x4
#define	DTX	0x2
#define	DRX	0x1

/* CSR0.  l.rap points to to it most of the time */

#define	ERR0	0x8000
#define	BABL	0x4000
#define	CERR	0x2000
#define	MISS	0x1000
#define	MERR	0x800
#define	RINT	0x400
#define	TINT	0x200
#define	IDON	0x100
#define	INTR	0x80
#define	INEA	0x40
#define	RXON	0x20
#define	TXON	0x10
#define	TDMD	0x8
#define	STOP	0x4
#define	STRT	0x2
#define	INIT	0x1

/*  lance csr3   */

#define	BSWP	0x4
#define	ACON	0x2
#define	BCON	0x1


/*  flag bits from a buffer descriptor in the rcv/xmt rings */

#define	OWN	0x8000	/* 1 means that the buffer can be used by the chip */
#define ERR	0x4000	/* erro summary, the OR of all error bits */
#define	FRAM	0x2000	/* CRC error and incoming packet not a multiple of 8 bits */
#define	OFLO	0x1000	/* (recv) lost some of the packet */
#define	MORE	0x1000	/* (trans) more than 1 retry to send the packet */
#define	CRC	0x800	/* (recv) crc error reading packet */
#define	ONE	0x800	/* (trans) one retry to transmit the packet */
#define	BUF	0x400	/* (recv) out of buffers while reading a packet */
#define	DEF	0x400	/* (trans) deffered while transmitting packet */
#define	STP	0x200	/* start of packet */
#define	ENP	0x100	/* end of packet */

/* cntflags bits from a buffer descriptor in the rcv/xmt rings */

#define	BUFF	0x8000	/* buffer error (host screwed up?) */
#define	UFLO	0x4000	/* underflow from memory */
#define	LCOL	0x1000	/* late collision (ether too long?) */
#define	LCAR	0x800	/* loss of carrier (ether broken?) */
#define	RTRY	0x400	/* couldn't transmit (bad station on ether?) */
#define	TTDR	0x3FF	/* time domain reflectometer */

/* 
 *   Some macros for dealing with lance memory addresses.  The lance splits
 *   its 24 vbit addresses across two 16 bit registers.
 */

#define	HADDR(a) (((u_long)(a)) >> 16 & 0xff)
#define	LADDR(a) (((u_long)a) & 0xffff)

/*
 *   Initialization block, this must be in RAM accessible by the lance.
 *   It is used to configure the lance on startup and is ignored after
 *   that.
 */

typedef struct initblock {
	u_short	mode;		/* chip control (see below) */
	u_short	etheraddr[3];	/* the ethernet physical address */
	u_short multi[4];	/* multicast addresses, 1 bit for each of 64 */
	u_short	rdralow;	/* receive buffer ring */
	u_short	rdrahigh;	/* (top three bits define size of ring) */
	u_short	tdralow;	/* transmit buffer ring */
	u_short	tdrahigh;	/* (top three bits define size of ring) */
} Initblock;

/*
 *   Communication with the lance is via a transmit and receive ring of
 *   message descriptors.  The Initblock contains pointers to and sizes of
 *   these rings.  The rings must be in RAM addressible by the lance.
 */

typedef	struct {
	u_short	laddr;		/* low order piece of address */
	u_short	flags;		/* flags and high order piece of address */
	short	size;		/* size of buffer */
	u_short	cntflags;	/* (rcv)count of bytes in buffer; (xmt) more flags */
} Msg;


/*
 *   Ethernet packet buffers.  These must also be in alnce addressible RAM.
 */

typedef	struct {
	u_char	d[6];
	u_char	s[6];
	u_short	type;
	u_char	data[1500];
	u_char	crc[4];
} Etherp;

typedef	struct {
	u_char	d[6];
	u_char	s[6];
	u_char	type[2];
	u_char	data[60];
} MinEtherp;



/*	Stuff queues are made of.  Looks a lot like dmr streams */

typedef	struct {
	struct	queue	*q;
	u_short		flag;	/* 0=empty, 1=open, 2=ready to use */
	u_short		type;	/* ethernet type */
	struct	block	*outb;	/* buffer for output */
} Ethertype;

/*
 * Lance state
 */

typedef	struct {
	Initblock	*li;		/* init block */
	u_char		ea[6];		/* our ether addr */
	u_short		*rap;		/* lance address register */
	u_short		*rdp;		/* lance data register */
	Msg		*rring;		/* start of receive ring */
	u_short		rl;		/* first rcv message belonging to lance */
	u_short		rc;		/* first rcv messgae belonging to cpu */
	struct	block	*rbp[NRRB];	/* receive buffers */
	Msg		*tring;		/* start of xmt ring */
	u_short		tl;		/* first xmt message belonging to lance */
	u_short		tc;		/* first xmt message belonging to cpu */
	struct	block	*tbp[NTRB];	/* transmitt buffers */
	struct	block	blocks[NRBLOCKS];/* all ether blocks */
	struct block	*free;		/* free list of lance blocks */
	Ethertype	e[NTYPES];
} Lance;


/*
 *  lanceram must be addressable by the lance chip.
 *  on the 147 this is just local ram.  Other interfaces
 *  could require this to be mapped on to a board.
 */

struct	lanceram	{
	Initblock	initblock;
	Msg		rring[NRRB];
	Msg		tring[NTRB];
	Etherp		e[NRBLOCKS];
};

char	*fcore	= "/dev/kmem";
char	*fnlist	= "/unix";
int	fc;

struct	nlist setup[] = {
#define	SL	0
	{ "_l" },
	0
};


int	allflag;
int	nflag;

main(argc, argv)
char **argv;
{

	while (--argc && **++argv == '-') {
		while (*++*argv)
		switch (**argv) {
		case 'a':
			allflag++;
			break;
		case 'n':
			nflag++;
			break;
		}
	}
	if (argc>0)
		fcore = argv[0];
	if ((fc = open(fcore, 0)) < 0) {
		printf("Can't find %s\n", fcore);
		exit(1);
	}
	if (argc>1)
		fnlist = argv[1];
	nlist(fnlist, setup);
	if (setup[0].n_type == -1) {
		printf("no namelist\n");
		exit(1);
	}
	enstat();
}

enstat()
{
	Lance l;
	register int nin, loc;
	register Ethertype *ep;
	register u_char *cp;

	nin = 0;
	printf("%08x\n", (long)setup[SL].n_value);
	setup[SL].n_value = 0x800b9480;
	kseek(fc, (long)setup[SL].n_value, 0);
	read(fc, (char *)&l, sizeof l);
	for (cp = l.ea; cp < &l.ea[6]; cp++)
		printf("%02x%c", *cp, cp == &l.ea[5] ? ' ' : ':');
	printf("\n");
	for (ep = l.e; ep < &l.e[NTYPES]; ep++) {
		printf("%02d: ", ep - l.e);
		printf("#%08x %d #%04x\n", ep->q, ep->flag, ep->type);
	}
}

putf(v, n)
{
	printf("%c", v ? n : ' ');
}

kseek(fd, offset, how)
{
	return lseek(fd, offset & ~0xC0000000, how);
}
