/* pcc.c:  Control code for the mvme147 peripheral channel controller chip */

#include "param.h"
#include "types.h"
#include "reboot.h"

#define	PCC	((struct pcc *) 0xfffe1000)

struct	pcc	{
	u_long	dma_ta;		/* dma table address (bits 1 and 0 are 0) */
	u_long	dma_da;		/* dma data address */
	u_long	dma_bc;		/* link, DFC2-0 and byte count */
	u_long	dma_hr;		/* data holding register */
	u_short	timer1p;	/* timer 1 preload */
	u_short	timer1c;	/* timer 1 count */
	u_short	timer2p;	/* timer 2 preload */
	u_short timer2c;	/* timer 2 count */
	u_char	timer1_icr;	/* timer 1 interrupt control register */
	u_char	timer1_csr;	/* timer 1 control status register */
	u_char	timer2_icr;	/* timer 2 interrupt control register */
	u_char	timer2_csr;	/* timer 2 control status register */
	u_char	acfail_icr;	/* ACFAIL interrupt control register */
	u_char	watchdog;	/* watchdog timer control register */
	u_char	printer_icr;	/* printer interrupt control register */
	u_char	printer_cr;	/* printer control register */
	u_char	dma_icr;	/* dma interrupt control register */
	u_char	dma_csr;	/* dma control status register */
	u_char	berr_icr;	/* bus error interrupt control register */
	u_char	dma_sr;		/* dma status register */
	u_char	abort_icr;	/* abort interrupt control register */
	u_char	dma_fcr;	/* dma function control register */
	u_char	serial_icr;	/* serial interrupt control register */
	u_char	pcc_cr;		/* pcc control register */
	u_char	lan_cr;		/* LANCE control register */
	u_char	pcc_sr;		/* pcc status register */
	u_char	scsi_icr;	/* scsi interrupt control register */
	u_char	ram_base;	/* ram base */
	u_char	si1_cr;		/* soft interrupt 1 control register */
	u_char	int_vbase;	/* interrupt vector base */
	u_char	si2_cr;		/* soft interrupt 2 control register */
	u_char	chip_rev;	/* revision of pcc */
	u_char	printer_dr;	/* printer data register */
	u_char	printer_sr;	/* printer status register */
};


pccinit()		/* set up interrupt vectors */
{
	PCC->int_vbase = 64;	/* beginning of interrupt region */
	PCC->si1_cr = 002;	/* level 2 interrupts for soft intr */
	PCC->serial_icr = 003;	/* level 3 (tty) interrupts are for ttys */
	PCC->scsi_icr = 014;	/* level 4 (bio) interrupts from disc and tape */
	PCC->lan_cr = 005;	/* level 5 is imp */
	PCC->timer1_icr = 006;	/* level 6 (clock) interrupts for clock */
	PCC->pcc_cr = 020;	/* enable interrupts */
	PCC->watchdog = 0xF5;	/* longest time, rst, enable */
}

static	int waiting = 0;

setdma(addr, count, in)	/* setup for dma transfer */
	caddr_t addr;
{
	PCC->dma_csr = 0;
	PCC->dma_da = (u_long)addr;
	PCC->dma_da = (u_long)addr & 0x00ffffff;	/* snip high bits */
	PCC->dma_bc = 0x05000000 | (count & 0xffffff);
	PCC->dma_csr = (in ? 0 : 4) | 1;
}

clrdma()		/* turn off the dma */
{
	PCC->dma_csr = 0;
	PCC->dma_da = 0;
	PCC->dma_bc = 0;
}

delay(n)
{
	PCC->timer2_csr = 0;	/* turn off timer */
	PCC->timer2p = -n;	/* number of 6.7us clicks to wait */
	PCC->timer2_csr = 7;	/* start counting */
	while ((PCC->timer2_csr >> 4) == 0)
		; 		/* wait for the timer to roll */
	PCC->timer2_csr = 0;	/* turn off */
}

clkstart()			/* crank the clock up */
{
	PCC->timer1_csr = 0;	/* turn off timer */
	PCC->timer1p = 62869;	/* 16.6667 tick */
	PCC->timer1_icr |= 010;	/* turn on the interrupts */
	PCC->timer1_csr = 3;	/* go */
}

clkreld(n)		/* called every Hz */
{
	PCC->timer1_icr |= 0200;	/* clear interrupt */
	PCC->watchdog = 0xF7;		/* hit the clr */
	return 0;
}

resetscsi(flag)			/* set scsi bus reset line */
{
	if (flag)
		PCC->scsi_icr |= 0x10;
	else
		PCC->scsi_icr &= ~0x10;
}


pcc_tty_intr_on()		/* turn on the ints for the serial chips */
{
	PCC->serial_icr |= 010;	/* turn on tty interrupts */
}

pccabort()			/* catch the black button */
{
	PCC->abort_icr = 0;	/* disable the interrupts */
	spl0();
	snap();			/* print the registers */
	panic("Black button");
}

pcc_lan_init()		/* setup lan part of pcc */
{
	PCC->lan_cr |= 010;	/* turn on interrupts */
}

pcc_set_si1()
{
	PCC->si1_cr |= 010;
}

pcc_reset_si1()
{
	PCC->si1_cr &= ~010;
}

pcc_reboot(flag)		/* shuter down Klem, she's a'pumpin mud */
{
	if (flag & RB_HALT)
		PCC->watchdog = 0;	/* WDClr */
	splhigh();
	for (;;)
		asm("stop #0x2700");	/* you are in a deep deep sleep */
}
