/* scsi.c:  Driver for wd3393 scsi controller chip */
/*	Scsi() will queue one request if busy and issue it after
 *	the interrupt handler for the current request is called
 *	and hasn't made another request.  This means that chains
 *	of transactions don't get broken.
 */


#include "param.h"
#include "types.h"
#include "buf.h"
#include "reboot.h"

#define	SCSIADDR	((u_char *) 0xfffe4000)
#define	SCSIDATA	((u_char *) 0xfffe4001)

#define OWN_ID		(0x00)
#define CTL_REG		(0x01)
#define TIMEOUT		(0x02)
#define TOTAL_SECS	(0x03)
#define TOTAL_HEADS	(0x04)
#define TOTAL_CYL_MSB 	(0x05)
#define TOTAL_CYL_LSB	(0x06)
#define LOG_ADDR_MSB	(0x07)
#define LOG_ADDR_2	(0x08)
#define	LOG_ADDR_3	(0x09)
#define LOG_ADDR_LSB	(0x0a)
#define SEC_NUM		(0x0b)
#define HEAD_NUM	(0x0c)
#define CYL_NUM_MSB	(0x0d)
#define CYL_NUM_LSB	(0x0e)
#define TARGET_LUN	(0x0f)
#define CMD_PHASE	(0x10)
#define SYN_XFER	(0x11)
#define XFER_CNT_MSB	(0x12)
#define XFER_CNT_2	(0x13)
#define XFER_CNT_LSB	(0x14)
#define DEST_ID		(0x15)
#define SRC_ID		(0x16)
#define SCSI_STATUS	(0x17)
#define CMD_REG		(0x18)
#define DATA_REG	(0x19)
#define AUX_STATUS	(0x1f)

#define CDB_SIZE	(0x00)
#define CDB_1		(0x03)
#define CDB_2		(0x04)
#define CDB_3		(0x05)
#define CDB_4		(0x06)
#define CDB_5		(0x07)
#define CDB_6		(0x08)
#define CDB_7		(0x09)
#define CDB_8		(0x0a)
#define CDB_9		(0x0b)
#define CDB_10		(0x0c)
#define CDB_11		(0x0d)
#define CDB_12		(0x0e)

#define STATUS_MASK	(0xf0)
#define CODE_MASK	(0x0f)
#define MCI_MASK	(0x07)

#define GET_STATUS(stat)	(stat & STATUS_MASK)
#define GET_CODE(stat)		(stat & CODE_MASK)
#define GET_MCI(stat)		(stat & MCI_MASK)

#define MCI_DATA_OUT		(0x00)
#define MCI_DATA_IN		(0x01)
#define MCI_CMD			(0x02)
#define MCI_STATUS		(0x03)
#define MCI_UNSP_INFO_OUT	(0x04)
#define MCI_UNSP_INFO_IN	(0x05)
#define MCI_MSG_OUT		(0x06)
#define MCI_MSG_IN		(0x07)

#define STATUS_RESET	(0x00)
#define STATUS_SUCCESS	(0x10)
#define STATUS_PAUSED	(0x20)
#define STATUS_TERM	(0x40)
#define STATUS_SRVC	(0x80)

#define RESET_NOADV	(STATUS_RESET | 0x00)
#define RESET_ADV	(STATUS_RESET | 0x01)

#define SUCC_RESELECT	(STATUS_SUCCESS | 0x0)
#define SUCC_SELECT	(STATUS_SUCCESS | 0x1)
#define SUCC_NOATN	(STATUS_SUCCESS | 0x3)
#define SUCC_ATN	(STATUS_SUCCESS | 0x4)
#define SUCC_XLAT_ADDR	(STATUS_SUCCESS | 0x5)
#define SUCC_SEL_XFER	(STATUS_SUCCESS | 0x6)

#define PAUSED_XFER_INFO	(STATUS_PAUSED | 0x0)
#define PAUSED_SAVE_PTRS	(STATUS_PAUSED | 0x1)
#define ABORT_SEL_RESEL		(STATUS_PAUSED | 0x2)
#define ABORT_SND_RCV_NOATN	(STATUS_PAUSED | 0x3)
#define ABORT_SND_RCV_ATN	(STATUS_PAUSED | 0x4)
#define ABORT_ABORTED		(STATUS_PAUSED | 0x5)
#define PAUSED_BAD_SCSI_ID	(STATUS_PAUSED | 0x7)

#define TERM_INVALID_CMD	(STATUS_TERM | 0x0)
#define TERM_UNXPCTD_DISC	(STATUS_TERM | 0x1)
#define TERM_TIMEOUT		(STATUS_TERM | 0x2)
#define TERM_PARITY_NOATN	(STATUS_TERM | 0x3)
#define TERM_PARITY_ATN		(STATUS_TERM | 0x4)
#define TERM_TOO_BIG		(STATUS_TERM | 0x5)
#define TERM_TARGET_MISMATCH	(STATUS_TERM | 0x6)
#define TERM_BAD_STATUS		(STATUS_TERM | 0x7)

#define SRVC_RESEL		(STATUS_SRVC | 0x0)
#define SRVC_RESEL_ADV		(STATUS_SRVC | 0x1)
#define SRVC_SEL_NOATN		(STATUS_SRVC | 0x2)
#define SRVC_SEL_ATN		(STATUS_SRVC | 0x3)
#define SRVC_ATN		(STATUS_SRVC | 0x4)
#define SRVC_DISC		(STATUS_SRVC | 0x5)
#define SRVC_WAIT_RCV		(STATUS_SRVC | 0x7)

#define CMD_RESET		(0x00)
#define CMD_ABORT		(0x01)
#define CMD_ATN			(0x02)
#define CMD_NACK		(0x03)
#define CMD_DISC		(0x04)
#define CMD_RESELECT		(0x05)
#define CMD_SEL_ATN		(0x06)
#define CMD_SEL_NOATN		(0x07)
#define CMD_SEL_XFER_ATN	(0x08)
#define CMD_SEL_XFER_NOATN	(0x09)
#define CMD_RESEL_RCV		(0x0a)
#define CMD_RESEL_SND		(0x0b)
#define CMD_WAIT_RCV		(0x0c)
#define CMD_SND_STAT_CMPLT	(0x0d)
#define CMD_SND_DISC		(0x0e)
#define CMD_SET_IDI		(0x0f)
#define CMD_RCV_CMD		(0x10)
#define CMD_RCV_DATA		(0x11)
#define CMD_RCV_MSG_OUT		(0x12)
#define CMD_RCV_UNSP_INFO_OUT	(0x13)
#define CMD_SND_STATUS		(0x14)
#define CMD_SND_DATA		(0x15)
#define CMD_SND_MSG_IN		(0x16)
#define CMD_SND_UNSP_INFO_IN	(0x17)
#define CMD_XLAT_ADDR		(0x18)
#define CMD_XFER_INFO		(0x20)
				 
#define	SBT		0x80

#define AUX_DBR		(0x01)
#define AUX_PE		(0x02)
#define AUX_CIP		(0x10)
#define AUX_BSY		(0x20)
#define AUX_LCI		(0x40)
#define AUX_INT		(0x80)

#define CTL_HSP		(0x01)
#define CTL_HA		(0x02)
#define CTL_IDI		(0x04)
#define CTL_EDI		(0x08)
#define CTL_HHP		(0x10)
#define CTL_DMA0	(0x00)
#define CTL_DMA1	(0x20)
#define CTL_DMA2	(0x40)
#define CTL_DMA3	(0x80)

#define SYN_XFER_VALUE	(0x00)

#define	get(a)		(*SCSIADDR = (a), *SCSIDATA)
#define	set(a, d)	(*SCSIADDR = (a), *SCSIDATA = d)
#define test_reg(reg, mask)	(get(reg) & (mask))
#define wait_off(reg, mask)	while (test_reg(reg, mask) != 0)
#define wait_on(reg, mask)	while (test_reg(reg, mask) == 0)
#define WAIT_7USECS	delay(2)				 

static
struct	wd3393	{
	u_char	control;		/* enable/disable certain functions */
	u_char	timeout;		/* = (Tper * Ficlk) / 80 */
	u_char	cdb[12];		/* the command to send */
	u_char	target_lun;		/* set during reconnect */
	u_char	cmd_phase;		/* where the chip is in things */
	long	xfer_cnt;		/* bytes to move */
	u_char	dest_id;		/* scsi destination register */
	u_char	src_id;			/* who am i */
	u_char	status;			/* chip status */
	u_char	command;		/* chip command */
} d;

static	busy;
static	next_waiting;
static	int next_id, next_flags, next_count;
static	char next_cdb[12];
static	caddr_t	next_addr;
static	int	(*next_intr)();

static	int	(*inthandler)();	/* upper level int handler */

static char	request_sense[12] = {0x03, 0, 0, 0, 0, 0};
static int	req_sense_status;

static char	unit_ready[12] = {0x00, 0, 0, 0, 0, 0};
static int	unit_ready_status;

extern u_char 	scsi_get_byte();
extern long	scsi_get_xfer_cnt();
extern int	scsi_req_sense_intr();
extern int	scsi_unit_ready_intr();

scsi(id, flags, cdb, addr, count, intfunc)		/* do scsi cmd */
	char *cdb;
	caddr_t addr;
	int (*intfunc)();
{
	register u_char *fp;
	register s, i;
	static inited = 0;

	s = splbio();
	if (busy) {
		next_id = id;
		next_flags = flags;
		bcopy(cdb, next_cdb, 12);
		next_addr = addr;
		next_count = count;
		next_intr = intfunc;
		next_waiting = 1;
		splx(s);
		return;
	} else
		busy = 1;
	splx(s);
	if (! inited) {		/* this never happens as on interrupt */
		resetscsi(1);			/* assert reset line */
		delay(100);
		resetscsi(0);			/* de-assert scsi reset line */
		delay(2000);
		set(OWN_ID, 7);			/* scsi id 7, no par, 10MHz */
		delay(2);
		set(CMD_REG, CMD_RESET);	/* reset the chip */
		sleep(&inthandler, PRIBIO);	/* okay to sleep */
		inited = 1;
	}
	inthandler = intfunc;
	if (count) 
		setdma(addr, count, (flags & B_READ));
	d.control = CTL_DMA3 | CTL_EDI;
	d.timeout = 62;		/* 500 ms */
	bcopy(cdb, d.cdb, sizeof d.cdb);
	if (id == 4)
		d.target_lun = 0;
	d.cmd_phase = 0;
	d.xfer_cnt = (count & 0xffffff) | (SYN_XFER_VALUE << 24);
	d.dest_id = id;
	d.status = 0;
	d.command = CMD_SEL_XFER_NOATN;		/* select and transfer w/o atn */
	WAIT_7USECS;
	*SCSIADDR = CTL_REG;		/* start with the control register */
	fp = (u_char *)&d;
	for (i = 0; i < ((char *)&d.command - (char *)&d)+1; i++)
		*SCSIDATA = *fp++;
}

pccscsiportintr()		/* catch the chip's interrupt */
{
	register	status;
	register	scsi_status;
	long		resid;
	
	status = get(SCSI_STATUS);
	switch (status) {
	case RESET_NOADV:				/* reset */
	case RESET_ADV:
		wakeup(&inthandler);
		break;
	case SUCC_SEL_XFER:		/* sel&xfer complete */
		scsi_status = get(TARGET_LUN);
		resid = scsi_get_xfer_cnt();
		scsi_finish_req(scsi_status, resid);
		break;
	default:
		scsi_exception(status);
		break;
	}
}

scsi_exception(stat)
int	stat;
{
	int	status;
	int	code;
	int	mci;
	
	status	= GET_STATUS(stat);
	code	= GET_CODE(stat);
	mci	= GET_MCI(stat);

	switch (status)
	{
	case STATUS_TERM:
		scsi_term_intr(stat);
		break;

	default:
		scsi_reboot(stat);
		break;
	}
}

scsi_term_intr(stat)
int	stat;
{
	int	status;
	int	code;
	int	mci;
	u_char	scsi_status;
	u_char	bus_status;
	u_char	msg_in;
	long	resid;
	
	status	= GET_STATUS(stat);
	code	= GET_CODE(stat);
	mci	= GET_MCI(stat);

	if (code & 0x8) {
		switch (mci) {
		case MCI_STATUS:
			resid = scsi_get_xfer_cnt();
			set(CTL_REG, CTL_EDI); /* Turn off the DMA */
			
			/* Get the scsi bus status byte from drive */
			bus_status = scsi_get_byte(&scsi_status);

			/* Get the command complete msg */
			msg_in = scsi_get_byte(&scsi_status);

			scsi_set_command(CMD_NACK);
			wait_off(AUX_STATUS, AUX_CIP);
			wait_on(AUX_STATUS, AUX_INT);
			scsi_status = get(SCSI_STATUS);

			scsi_finish_req(bus_status, resid);
			break;
		default:
			scsi_reboot(stat);
			break;
		}
	}
	else if (stat == TERM_TIMEOUT) {
		resid = scsi_get_xfer_cnt();
		scsi_finish_req(0x100, resid);
	}
	else {
		scsi_reboot(stat);
	}
}

scsi_reboot(status)
int	status;
{
	printf("pccscsiintr: status 0x%x\n", status);
	pcc_reboot(RB_NOSYNC);
}

scsi_finish_req(status, resid)
int	status;
long	resid;
{
	register 	(*handler)();
	
	busy = 0;
	handler = inthandler;
	if (next_waiting) {
		next_waiting = 0;
		scsi(next_id, next_flags, next_cdb, next_addr, 
		     next_count, next_intr);
	}
	if (handler != NULL)
		(*handler)(status, resid);	/* call user int handler */
	else
		printf("scsi_finish_req: inthandler NULL!\n");
	
}

u_char scsi_get_byte(status)
u_char	*status;
{
	u_char	ret;

	*status = scsi_set_command(SBT | CMD_XFER_INFO); /* get status */
	wait_off(AUX_STATUS, AUX_CIP);
	wait_on(AUX_STATUS, AUX_DBR);
	ret = get(DATA_REG);
	wait_on(AUX_STATUS, AUX_INT);
	*status = get(SCSI_STATUS);
	return(ret);
}

scsi_set_command(command)
u_char	command;
{
	int	scsi_status;

	scsi_status = 0;
	wait_off(AUX_STATUS, AUX_CIP);
	do {
		if (test_reg(AUX_STATUS, AUX_INT) != 0) {
			scsi_status = get(SCSI_STATUS);
		}
		set(CMD_REG, command);
	} while (test_reg(AUX_STATUS, AUX_LCI) != 0);

	return(scsi_status);
}
		
scsi_get_xfer_cnt()
{
	int	ret;
	u_char	cnt[4];

	cnt[0] = 0;
	cnt[1] = get(XFER_CNT_MSB);
	cnt[2] = get(XFER_CNT_2);
	cnt[3] = get(XFER_CNT_LSB);

	bcopy(cnt, &ret, 4);

	return(ret);
}

scsi_req_sense(id, addr, len)
int	 id;
caddr_t	*addr;
u_char	 len;
{
	request_sense[4] = len;
	scsi(id, B_READ, request_sense, addr, len, scsi_req_sense_intr);
	sleep(request_sense);
	return(req_sense_status);
}

static scsi_req_sense_intr(status, resid)
int	status;
long	resid;
{
	req_sense_status = status;
	wakeup(request_sense);
}

scsi_unit_ready(id)
int	id;
{
	scsi(id, B_READ, unit_ready, NULL, 0, scsi_unit_ready_intr);
	sleep(unit_ready);
	return(unit_ready_status);
}

static scsi_unit_ready_intr(status, resid)
int	status;
long	resid;
{
	unit_ready_status = status;
	wakeup(unit_ready);
}


