/* internet protocol line discipline  */

#include "param.h"
#include "types.h"
#include "errno.h"
#include "in.h"
#include "inio.h"
#include "ip_var.h"
#include "stream.h"
#include "ethernet.h"

/*
 * definition for extern linkage to queue stuff.
 * That is, just the ipldqinit struct is visable to the rest
 * of the world.
 */

static	upsrv(), qopen(), qclose(), dnsrv();
extern	putq(), nullsys();

struct	qinit	ipldqinit[] = {
/*up*/	{ putq, upsrv, qopen, qclose, 2*1520, 20 },
/*dn*/	{ putq, dnsrv, nullsys, nullsys, 2*1520, 20}
};

/* parameters to ipld */

#define	MINIPSIZE	20
#define	IP_DEFAULT_MTU	1500
#define	P_FORWARD	2		/* ip dev to send forwards to */

/*
 * the ipif table is shared with the ip code, but lives here.
 */

struct	ipif	ipif[NIPLD];		/* the interface table */

static	struct	arp	arptab[NARP];	/* arp resolve table */


#define	bclear(bp)	((bp)->rptr = (bp)->wptr = (bp)->base,\
				bp->type = M_IOCACK)


static
qopen(dev, q)		/* open up queue */
	dev_t dev;
	struct queue *q;
{
	register struct ipif *p;

	for (p = ipif; p < &ipif[NIPLD]; p++)
		if (p->queue == NULL) {
			p->queue = OTHERQ(q);
			p->flags = IP_DIDLE|IP_UIDLE;
			p->mtu = IP_DEFAULT_MTU;
			p->thishost = 0;
			p->that = 0;
			p->mask = 0;
			p->broadcast = 0;
			p->ipackets = p->ierrors = 0;
			p->opackets = p->oerrors = 0;
			p->arp = 0;	/* really current protocol */
/*
			p->dev = dev;
*/
			q->ptr = (char *)p;
			OTHERQ(q)->ptr = (char *)p;
			return 0;
		}
	return ENXIO;
}

static
qclose(q)		/* shut a queue down */
	struct queue *q;
{
	((struct ipif *)q->ptr)->queue = NULL;
}

/*
 * The upsrv routine processes blocks that contain ip frames.
 * It looks for an ip device to put the block on.
 */

static
upsrv(q)		/* ip upserver */
	register struct queue *q;
{
	register struct iphdr *ip;
	register struct block *bp;
	register struct ipif *ifp;

	ifp = (struct ipif *)q->ptr;
	while (q->first) {		/* get rid of all blocks */
		bp = getq(q);
		if (bp->type != M_DATA) {
			(*q->next->qinfo->putp)(q->next, bp);
			continue;
		}
		ipld_input(ifp, bp);
	}
}

static
dnsrv(q)		/* service down stream messages */
	register struct queue *q;
{
	register struct ipif *ifp;

	ifp = (struct ipif *)q->ptr;
	while (q->first && (q->next->flag & QFULL) == 0) {
		if (q->first->type != M_IOCTL) {
			(*q->next->qinfo->putp)(q->next, getq(q));
			continue;
		}
		ipld_ioctl(q, ifp, getq(q));
	}
	if (q->first)
		q->next->flag |= QWANTW;
}

static
ipld_ioctl(q, p, bp)	/* process ioctl */
	register struct queue *q;
	register struct ipif *p;
	register struct block *bp;
{
	struct route route;
	struct arp a;
	register union ifs *ifsp;
	in_addr dmask();

	switch (stiocom(bp)) {
	case IPIOLOCAL:
		bcopy(stiodata(bp), (char *)&p->thishost, sizeof (in_addr));
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIOHOST:
		bcopy(stiodata(bp), (char *)&p->that, sizeof (in_addr));
		bclear(bp);
		qreply(q, bp);
		if (p->mask == 0)
			p->mask = dmask(p->that);
		break;
	case IPIONET:
		bcopy(stiodata(bp), (char *)&p->that, sizeof (in_addr));
		bclear(bp);
		qreply(q, bp);
		if (p->mask == 0)
			p->mask = dmask(p->that);
		break;
	case IPIOMASK:
		bcopy(stiodata(bp), (char *)&p->mask, sizeof (in_addr));
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIOMTU:
		bcopy(stiodata(bp), (char *)&p->mtu, sizeof (int));
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIOARP:
		p->flags |= IP_ARP;
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIORESOLVE:
		bcopy(stiodata(bp), (char *)&a, sizeof (struct arp));
		arp_install(a.inaddr, a.enaddr);
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIOROUTE:
		bcopy(stiodata(bp), (char *)&route, sizeof (route));
		rte_install(route);
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIOGETIFS:
		ifsp = (union ifs *)stiodata(bp);
		if (ifsp->index < 0 || ifsp->index > NIPLD) {
			bclear(bp);
			bp->type = M_IOCNAK;
			qreply(q, bp);
		}
		bcopy((char *)&ipif[ifsp->index], 
			bp->rptr,
			sizeof(union ifs));
		bp->wptr = bp->rptr + sizeof (union ifs);
		bp->type = M_IOCACK;
		qreply(q, bp);
		break;
	default:
		(*q->next->qinfo->putp)(q->next, bp);
		break;
	}
}



static
ipld_input(ifp, bp)	/* process an internet pracket */
	register struct ipif *ifp;
	register struct block *bp;
{
	register hlen;
	register struct iphdr *p;
	register delim;

	delim = bp->class & S_DELIM;
	if (ifp->flags & IP_UIDLE) {
		if (ifp->flags & IP_ARP)
			bp->rptr += 14;
		ifp->ipackets++;
		p = (struct iphdr *)bp->rptr;
		if ((p->ver_ihl & 0xf0) != 0x40)
			goto bad;
		if ((hlen = (p->ver_ihl & 0x0f) << 2) < MINIPSIZE)
			goto bad;
		if (p->chksum = cksum((char *)p, hlen))
			goto bad;
		ifp->arp = p->proto;
		if (p->dest == 0xffffffff)
			goto good;
		if ((p->dest & ifp->mask) == ifp->that)
		if ((p->dest | ifp->mask) == 0xffffffff)
			goto good;
		if (p->dest == ifp->thishost)
			goto good;
		if (p->dest == (ifp->thishost & ~1))
			goto good;
		if ((p->dest & ~ifp->mask) == 0)
			goto bad;
		ifp->arp = P_FORWARD;
good:
		/* do options here */
		if (p->fragnflag & ~IP_MORE_FRAG)
			/* do reassembly */ goto bad;
		if (!delim)		/* first of a set */
			ipld_set(ifp, IP_UIDLE, IP_UPACKET);
		ip_upqueue(ifp->arp, bp);
		return;
	} else if (ifp->flags & IP_UTOSSING) {
		freeb(bp);
		if (delim)
			ipld_set(ifp, IP_UTOSSING, IP_UIDLE);
		return;
	} else if (ifp->flags & IP_UPACKET) {
		ip_upqueue(ifp->arp, bp);
		if (delim)
			ipld_set(ifp, IP_UPACKET, IP_UIDLE);
		return;
	}
bad:
	if (!delim)
		ipld_set(ifp, IP_UIDLE, IP_UTOSSING);
	ifp->ierrors++;
	freeb(bp);
}

/*
 * This code inserts an ethernet header on the front of a frame
 * if the arp flag is set.  The collection of blocks that form the
 * packet should be delimited.
 * We also must toss entire packets if the input queue is full, so
 * we need to remember that we are tossing the current frame.
 * Another assumption is that the ipdev, which calls this routine,
 * will pass all of the frame with out getting preempted by another
 * ipdev.  That would cause the two frames to be interleaved.
 */

ipld_output(ifp, bp, gate)	/* process an outgoing ip packet */
	register struct ipif *ifp;
	register struct block *bp;
	register in_addr gate;		/* where to send this */
{
	register struct block *ebp;	/* enet header */
	register struct ether *ep;	/* overlay */
	register struct iphdr *p;
	register struct arp *ap;
	int delim;

	delim = bp->class & S_DELIM;
	if (ifp->flags & IP_DIDLE) {	/* start of packet */
		ifp->opackets++;
		if (ifp->queue->flag & QFULL) {
			freeb(bp);
			ifp->oerrors++;
			if (! delim)
				ipld_set(ifp, IP_DIDLE, IP_DTOSSING);
			return;
		}
		p = (struct iphdr *) bp->rptr;
		if ((ifp->flags & IP_ARP) == 0) {
			(*ifp->queue->qinfo->putp)(ifp->queue, bp);
			if (! delim)
				ipld_set(ifp, IP_DIDLE, IP_DINPACKET);
			return;
		}
		/* here means we stick a enet header on the front */
		ebp = allocb(14);
		ebp->wptr += 14;
		ep = (struct ether *)ebp->rptr;
		for (ap = arptab; ap < &arptab[NARP]; ap++)
			if (ap->inaddr == gate)
				break;
		if (ap == &arptab[NARP]) {	/* unknown address */
			bcopy((char *)&p->dest, ebp->rptr, sizeof (in_addr));
			ebp->wptr = ebp->rptr + sizeof (in_addr);
			ebp->class |= S_DELIM;
			reply(ifp->queue, ebp);
			freeb(bp);	/* toss unsendable -- no waiting */
			if (! delim)
				ipld_set(ifp, IP_DIDLE, IP_DTOSSING);
			ifp->oerrors++;
			return;
		}
		bcopy(ap->enaddr, ep->dhost, ETHERALEN);
		ep->type = 0x800;	/* ip */
		(*ifp->queue->qinfo->putp)(ifp->queue, ebp);
		(*ifp->queue->qinfo->putp)(ifp->queue, bp);
		if (! delim)
			ipld_set(ifp, IP_DIDLE, IP_DINPACKET);
		return;
	} else if (ifp->flags & IP_DTOSSING) {
		freeb(bp);
		if (!delim)
			return;
		ipld_set(ifp, IP_DTOSSING, IP_DIDLE);
		return;
	} else if (ifp->flags & IP_DINPACKET) {
		/* we only look at QFULL at the beginning of packts */
		(*ifp->queue->qinfo->putp)(ifp->queue, bp);
		if (! delim)
			return;
		ipld_set(ifp, IP_DINPACKET, IP_DIDLE);
		return;
	} else {
                printf("ipld_output:  can't happen: ifp->flags 0%o\n", ifp->flags);
		freeb(bp);
		return;
	}
}

static
in_addr
dmask(ipa)	/* generte default address mask */
	in_addr;
{
	if ((ipa & 0x80000000) == 0x00000000)
		return 0xff000000;
	if ((ipa & 0xc0000000) == 0x80000000)
		return 0xffff0000;
	if ((ipa & 0xe0000000) == 0xc0000000)
		return 0xffffff00;
	return 0;
}

arp_install(ip, enet)	/* look up 'ip' in cache */
	in_addr ip;
	u_char enet[6];
{
	register struct arp *p, *ap = NULL;

	for (p = arptab; p < &arptab[NARP]; p++) {
		if (p->inaddr == ip) {
			bcopy(enet, p->enaddr, sizeof p->enaddr);
			return;
		}
		if (ap == NULL && p->inaddr == NULL)
			ap = p;
	}
	ap->inaddr = ip;
	bcopy(enet, ap->enaddr, sizeof ap->enaddr);
}


arp_remove(ip)	/* remove arp entry */
	register in_addr ip;
{
	register struct arp *p;

	for (p = arptab; p < &arptab[NARP]; p++)
		if (p->inaddr == ip)
			p->inaddr = NULL;
}

pr_ifs(ifp)	/* print ifs structure */
	register struct ipif *ifp;
{
        printf("pr_ifs: 0x%x\n", ifp);
	printf("\tqueue 0x%x\n", ifp->queue);
	printf("\tflags 0%o\n", ifp->flags);
	printf("\tmtu %d\n", ifp->mtu);
	printf("\tthishost 0x%x\n", ifp->thishost);
	printf("\tthat 0x%x\n", ifp->that);
	printf("\tmask 0x%x\n", ifp->mask);
	printf("\tbraodcast 0x%x\n", ifp->broadcast);
	printf("\tipackets %d, ierrors %d\n", ifp->ipackets, ifp->ierrors);
	printf("\topackets %d, oerrors %d\n", ifp->opackets, ifp->oerrors);
	printf("\tarp %d\n", ifp->arp);
	printf("\tdev %d,%d\n", major(ifp->dev), minor(ifp->dev));
}

static
ipld_set(ifp, unset, set)
	register struct ipif *ifp;
	register set, unset;
{
	if (set == IP_UIDLE)
		ifp->arp = 0;
	ifp->flags |= set;
	ifp->flags &= ~unset;
}

/*
 * To notify the layers above that a message wasn't send
 * because there was not entry in the arp table, I need to 
 * find the matching queue and put the message on that queues`
 * down stream module.  This keeps this module from trying to 
 * treat the message as a ip datagram.
 */

static
reply(q, bp)
	register struct queue *q;
	register struct block *bp;
{
	register struct queue *nq;

	nq = OTHERQ(q)->next;
	(*nq->qinfo->putp)(nq, bp);
}
