/* cm.c: streams driver for ADAPTIVE NCM board */
#include "param.h"
#include "types.h"
#include "buf.h"
#include "errno.h"
#include "proc.h"
#include "inode.h"
#include "dir.h"
#include "file.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "signal.h"
#include "user.h"
#include "stream.h"
#include "map.h"
#include "enio.h"
#include "ethernet.h"

/* configuration */

#define NTYPES		16	/* max enet packet types (even power of 2 */
#define	NRRB		64	/* number of recieve buffers per Lance */
#define	LOGNRRB		6	/* log of NRRB*/
#define	NTRB		16	/* number of transmit buffers */
#define	LOGNTRB		4	/* log of NTRB */
#define	CMPRI		28	/* sleeping like a tty */
#define	SPLIMP		6	/* interrupt level */
#define LANCEAP		((u_short *)0xfe200002)
#define LANCEDP		((u_short *)0xfe200000)
#define	LANCERAM	((u_short *)0xfe000000)
#define	FIFO		((u_short *)0xfe100000)
#define	FIFO_ST		((u_short *)0xfe500000)
#define	NRBLOCKS	(2*(NTRB+NRRB))
#define LANCE1		0
#define LANCE2		1


#define	RSUCC(x)	(((x)+1)%NRRB)
#define	TSUCC(x)	(((x)+1)%NTRB)

#define lancenum(x) ((int)((minor(x))>>6))
/* macro to map an address into the no cache segment */


/*   mode bits in the lance initialization block */

#define	PROM	0x8000
#define	INTL	0x40
#define	DRTY	0x20
#define	COLL	0x10
#define	DTCR	0x8
#define	LOOP	0x4
#define	DTX	0x2
#define	DRX	0x1

/* CSR0.  l.rap points to to it most of the time */

#define	ERR0	0x8000
#define	BABL	0x4000
#define	CERR	0x2000
#define	MISS	0x1000
#define	MERR	0x800
#define	RINT	0x400
#define	TINT	0x200
#define	IDON	0x100
#define	INTR	0x80
#define	INEA	0x40
#define	RXON	0x20
#define	TXON	0x10
#define	TDMD	0x8
#define	STOP	0x4
#define	STRT	0x2
#define	INIT	0x1

/*  lance csr3   */

#define	BSWP	0x4
#define	ACON	0x2
#define	BCON	0x1


/*  flag bits from a buffer descriptor in the rcv/xmt rings */

#define	OWN	0x8000	/* 1 means that the buffer can be used by the chip */
#define ERR	0x4000	/* erro summary, the OR of all error bits */
#define	FRAM	0x2000	/* CRC error and incoming packet not a multiple of 8 bits */
#define	OFLO	0x1000	/* (recv) lost some of the packet */
#define	MORE	0x1000	/* (trans) more than 1 retry to send the packet */
#define	CRC	0x800	/* (recv) crc error reading packet */
#define	ONE	0x800	/* (trans) one retry to transmit the packet */
#define	BUF	0x400	/* (recv) out of buffers while reading a packet */
#define	DEF	0x400	/* (trans) deffered while transmitting packet */
#define	STP	0x200	/* start of packet */
#define	ENP	0x100	/* end of packet */

/* cntflags bits from a buffer descriptor in the rcv/xmt rings */

#define	BUFF	0x8000	/* buffer error (host screwed up?) */
#define	UFLO	0x4000	/* underflow from memory */
#define	LCOL	0x1000	/* late collision (ether too long?) */
#define	LCAR	0x800	/* loss of carrier (ether broken?) */
#define	RTRY	0x400	/* couldn't transmit (bad station on ether?) */
#define	TTDR	0x3FF	/* time domain reflectometer */

/* 
 *   Some macros for dealing with lance memory addresses.  The lance splits
 *   its 24 vbit addresses across two 16 bit registers.
 */

#define	HADDR(a) (((u_long)(a)) >> 16 & 0xff)
#define	LADDR(a) (((u_long)a) & 0xffff)

/*
 *   Initialization block, this must be in RAM accessible by the lance.
 *   It is used to configure the lance on startup and is ignored after
 *   that.
 */

typedef struct initblock {
	u_short	mode;		/* chip control (see below) */
	u_short	etheraddr[3];	/* the ethernet physical address */
	u_short multi[4];	/* multicast addresses, 1 bit for each of 64 */
	u_short	rdralow;	/* receive buffer ring */
	u_short	rdrahigh;	/* (top three bits define size of ring) */
	u_short	tdralow;	/* transmit buffer ring */
	u_short	tdrahigh;	/* (top three bits define size of ring) */
} Initblock;

/*
 *   Communication with the lance is via a transmit and receive ring of
 *   message descriptors.  The Initblock contains pointers to and sizes of
 *   these rings.  The rings must be in RAM addressible by the lance.
 */

typedef	struct {
	u_short	laddr;		/* low order piece of address */
	u_short	flags;		/* flags and high order piece of address */
	short	size;		/* size of buffer */
	u_short	cntflags;	/* (rcv)count of bytes in buffer; (xmt) more flags */
} Msg;


/*
 *   Ethernet packet buffers.  These must also be in alnce addressible RAM.
 */

typedef	struct {
	u_char	d[6];
	u_char	s[6];
	u_short	type;
	u_char	data[1500];
	u_char	crc[4];
} Etherp;

typedef	struct {
	u_char	d[6];
	u_char	s[6];
	u_char	type[2];
	u_char	data[60];
} MinEtherp;



/*	Stuff queues are made of.  Looks a lot like dmr streams */

typedef	struct {
	struct	queue	*q;
	u_short		flag;	/* 0=empty, 1=open, 2=ready to use */
	u_short		type;	/* ethernet type */
	struct	block	*outb;	/* buffer for output */
	caddr_t		ls;	/* a pointer to a lance structure */
	caddr_t		ep;	/* a pointer to the other lance's e entry */
	u_int		pseudo;	/* 1=cm2 pseudo lance 0=cm[01] lance */
} Ethertype;

/*
 * Lance state
 */

typedef	struct {
	Initblock	*li;		/* init block */
	u_char		ea[6];		/* our ether addr */
	u_short		*rap;		/* lance address register */
	u_short		*rdp;		/* lance data register */
	Msg		*rring;		/* start of receive ring */
	u_short		rl;		/* first rcv msg belonging to lance */
	u_short		rc;		/* first rcv messgae belonging to cpu */
	struct	block	*rbp[NRRB];	/* receive buffers */
	Msg		*tring;		/* start of xmt ring */
	u_short		tl;		/* first xmt msg belonging to lance */
	u_short		tc;		/* first xmt message belonging to cpu */
	struct	block	*tbp[NTRB];	/* transmitt buffers */
	Ethertype	e[NTYPES];
} Lance;

typedef struct {
	struct	block	blocks[NRBLOCKS];/* all ether blocks */
	struct block	*free;		/* free list of lance blocks */
} Lblk;

/*
 *  lanceram must be addressable by the lance chip.
 *  on the 147 this is just local ram.  Other interfaces
 *  could require this to be mapped on to a board.
 */

struct	lanceram	{
	Initblock	initblock[2];
	Msg		rring[2][NRRB];
	Msg		tring[2][NTRB];
	Etherp		e[NRBLOCKS];
};


static	Lance	l[2];
static 	Lblk	lb;
static	int	inpackets, outpackets;
static	int cmdnsrv(), cmqopen();
extern	int srv();
extern	int nulldev();
extern	int putq();

struct	qinit	cmqinit[] = {
/* up */{ putq, srv, cmqopen, nulldev, 1520, 20 },
/* dn */{ putq, cmdnsrv, nulldev, nulldev, 1520, 20 }
};
static	struct	block	*eallocb();

#define	DBREST	0001
#define	DBBLOCK	0002
#define DBSEND	0004

static	cmdebug = 0;
int	activenc = 0;	/* 0=standby NC  1=active NC */
int	complex = 0;	/* 0=undef:1=A side:2=B side */
int	activelan = 0;
static u_short actmcast[] = {0x2300, 0, 0xc000};
static u_short sbymcast[] = {0x2300, 0, 0xc001};


static
cmqopen(dev, q)		/* link to a queue provided by streams */
	dev_t dev;
	struct queue *q;
{
	Ethertype *e;
	if (lancenum(minor(dev)) == 2) {
		e = (Ethertype *)&l[0].e[minor(dev)&NTYPES-1];
		q->ptr = OTHERQ(q)->ptr = (caddr_t)e;
		e->pseudo = 1;
		e->ep = (caddr_t)&l[1].e[minor(dev)&NTYPES-1];
		e = (Ethertype *)&l[1].e[minor(dev)&NTYPES-1];
		e->pseudo = 1;
		e->ep = (caddr_t)&l[0].e[minor(dev)&NTYPES-1];
		l[0].e[minor(dev) & NTYPES-1].q = q;
		l[1].e[minor(dev) & NTYPES-1].q = q;
	} else {
		q->ptr = (char *)&l[lancenum(dev)].e[minor(dev) & (NTYPES-1)];
		OTHERQ(q)->ptr = (char *)&l[lancenum(dev)].e[minor(dev)&(NTYPES-1)];
		l[lancenum(dev)].e[minor(dev) & (NTYPES-1)].q = q;
		l[lancenum(dev)].e[minor(dev)&(NTYPES-1)].ls=(caddr_t)&l[lancenum(dev)];
	}
}

cmopen(dev, flag)
	dev_t dev;
{
	register Ethertype *e;
	static inited = 0;
	int i,j,k,lan;

	if (cmdebug & DBREST)
		printf("cmopen:top\n");
	if (minor(dev) < 0) {
	printf("dev out of range\n");
	printf("dev = %x  minor(dev) = %x\n", dev, minor(dev));
		u.u_error = ENXIO;
		return;
	}
	if (fushort(LANCERAM) == -1) {
		u.u_error = ENXIO;
		return;
	}
	if (complex == 0) {
		u.u_error = ENXIO;
		return;
	}
	if (!inited) {
		splimp();
		cminit();
		linit(LANCE1);
		linit(LANCE2);
		binit();
		inited = 1;
		spl0();
	}
	lan = lancenum(dev) < 2 ? lancenum(dev) : 0;
	if ((e = &l[lan].e[minor(dev) & NTYPES - 1])->flag) {
		u.u_error = ENXIO;
		return;
	}
	if (lancenum(dev) == 2) {
		j = 0;
		k = 1;
	} else
		j = k = lancenum(dev);
	for (i = j; i <= k; i++) {
		e = &l[i].e[minor(dev) & (NTYPES-1)];
		e->outb = NULL;
		e->flag = 1;
		e->type = 0;
	}
	if (cmdebug & DBREST)
		printf("cmopen:near bot\n");
}

cmclose(dev, flag)
	dev_t dev;
{
	register Ethertype *e;
	int i,j,k;

	if (cmdebug & DBREST)
		printf("cmclose: top\n");
	if (lancenum(dev) == 2) {
		j = 0;
		k = 1;
	} else
		j = k = lancenum(dev);
	for (i = j; i <= k; i++) {
		e = &l[i].e[minor(dev) & (NTYPES-1)];
		e->flag = 0;	/* closed */
		if (e->outb)
			efreeb(e->outb);
		e->q = NULL;
		e->type = 0;
	}
	if (cmdebug & DBREST)
		printf("cmclose: bot\n");
}

static
cmdnsrv(q)		/* down stream service procedure */
	register struct queue *q;
{
	register n;
	register struct block  *bp;
	register struct block *outb;
	register Lance *lnp, *tp;
	register Ethertype *e;

	if (cmdebug & DBREST)
		printf("cmdnsrv: top\n");
	e = (Ethertype *)q->ptr;
	if (e->pseudo == 1)
		lnp = &l[activelan & 1];
	else
		lnp = (Lance *)(((Ethertype *)(q->ptr))->ls);
	while(TSUCC(lnp->tc) != lnp->tl && (bp = getq(q))) {
		if (cmdebug & DBSEND) 
			cmprbp("cmdnsrv: from queue", bp);
		if (bp->type == M_IOCTL) {
			cmioctl(q, bp);
			continue;
		}
		if (bp->type != M_DATA) {
			freeb(bp);
			continue;
		}
		if ((outb = e->outb) == NULL)
			outb = e->outb = eallocb();
		n = min(outb->lim - outb->wptr, bp->wptr - bp->rptr);
		if (n > 0) {
			bcopy(bp->rptr, outb->wptr, n);
			bp->rptr += n;
			outb->wptr += n;
		} else
			bp->rptr = bp->wptr; 	/* toss rest */
		if (cmdebug & DBSEND)
			cmprbp("cmdnsrv: outb", outb);
		if (bp->class & S_DELIM) {
			if (cmdebug & DBREST)
				printpacket(outb->base, lnp);
			if (cmdebug & DBSEND)
				printf("calling put_tring\n");
			put_tring(outb, lnp);
			e->outb = NULL;
		}
		freeb(bp);
	}
	if (cmdebug & DBREST)
		printf("cmdnsrv: bot\n");
}

static
put_tring(bp, lp)
	register struct block *bp;
	register Lance *lp;
{
	Etherp *e;
	register n;

	e = (Etherp *) bp->base;
	n = bp->wptr - bp->rptr;
	bcopy(lp->ea, e->s, sizeof lp->ea);
	
	/* set up the ring descriptor */

	lp->tbp[lp->tc] = bp;
	lp->tring[lp->tc].size = -(n > 60 ? n : 60);
	lp->tring[lp->tc].cntflags = 0;
	lp->tring[lp->tc].laddr = LADDR(bp->base);
	lp->tring[lp->tc].flags = OWN | STP | ENP | HADDR(bp->base);
	if (cmdebug & DBREST)
		printf("put_tring:bp 0x%x, lp->tc %d\n", bp, lp->tc);
	lp->tc = TSUCC(lp->tc);
}




cmioctl(q, bp)	/* handle io control message */
	register struct queue *q;
	register struct block *bp;
{
	register Ethertype *e;
	register Lance *lnp;

	if (cmdebug & DBREST)
		printf("cmioctl:top\n");
	e = (Ethertype *)q->ptr;
	if (((Ethertype *)q->ptr)->pseudo == 1)
		lnp = &l[activelan & 1];
	else
		lnp = (Lance *)((Ethertype *)q->ptr)->ls;
	switch (stiocom(bp)) {
	case ENIOTYPE:		/* set the protocol type */
		bcopy(stiodata(bp), (caddr_t)&e->type, sizeof e->type);
		e->flag = 2;		/* okay to receive messages */
		if (((Ethertype *)q->ptr)->pseudo == 1) {
			e = (Ethertype *)e->ep;
			bcopy(stiodata(bp), (caddr_t)&e->type, sizeof e->type);
			e->flag = 2;		/* okay to receive messages */
		}
		bp->type = M_IOCACK;
		bp->rptr = bp->wptr = bp->base;
		qreply(q, bp);
		break;
	case ENIOADDR:		/* what about this */
		bcopy((caddr_t)lnp->ea, bp->rptr, sizeof lnp->ea);
		bp->type = M_IOCACK;
		bp->wptr = bp->rptr + sizeof lnp->ea;
		qreply(q, bp);
		break;
	default:
		bp->type = M_IOCNAK;
		bp->rptr = bp->wptr = bp->base;
		qreply(q, bp);
	}
}

cm2intr()
{
	if (cmdebug & DBREST)
		printf("cm2intr\n");
	cmint(LANCE2);
}

cmintr()
{
	if (cmdebug & DBREST)
		printf("cmintr\n");
	cmint(LANCE1);
}

cmint(lanceno)
int lanceno;
{
	u_short csr;
	int xrfreed = 0;
	Etherp *p;
	Ethertype *e;
	register t, size;
	struct block  *bp, *bp2;
	Lance *lp;

	lp = &l[lanceno & 1];
	*lp->rap = 0;
	csr = *lp->rdp;
 	if (cmdebug & DBREST)
		printf("cmintr top: csr %x\n", csr);

	/* turn off interrupt and any error indicators */
	
	*lp->rdp = IDON|INEA|TINT|RINT|BABL|CERR|MISS|MERR;

	/* see if an error occurred */

	if (*l->rdp & (BABL|MISS|MERR|CERR)) {
		*lp->rdp = BABL|CERR|MISS|MERR;
		printf("cm: err 0x%x\n", csr & (BABL|CERR|MISS|MERR));
	}
	if (csr & IDON)
		*lp->rdp = INEA|STRT;

	/* look for rcv'd packets, put in queue for type */

	for (; lp->rl!=lp->rc && (lp->rring[lp->rl].flags & OWN)==0; lp->rl=RSUCC(lp->rl)) {
		inpackets++;
		bp = lp->rbp[lp->rl];
		if (cmdebug & DBREST)
			printf("cmintr: bp 0x%x, lp->rl %d\n", bp, lp->rl);
		if (lp->rring[lp->rl].flags & ERR) {
#ifdef notdef
			printf("rcv error 0x%x\n",
				lp->rring[lp->rl].flags & (FRAM|OFLO|CRC|BUFF));
#endif
			efreeb(bp);
			continue;
		}

		p = (Etherp *)bp->rptr;

		if (memcmp((char *)actmcast,p->d,6) == 0)
		if (activenc == 0) {
			efreeb(bp);
			continue;
		}
		if (memcmp((char *)sbymcast,p->d,6) == 0)
		if (activenc != 0) {
			efreeb(bp);
			continue;
		}
		/* see if a queue exists for this packet type */
	
		if (cmdebug & DBREST)
			printpacket(p, lp);
		for (e = &lp->e[0]; e < &lp->e[NTYPES]; e++)
			if (e->flag == 2 && p->type == e->type)
				break;

 		/* 
		 * If no match, see if any stream has type -1.
		 * It matches all packtes.
		 */

		if (e == &lp->e[NTYPES]) {
			for (e = &lp->e[0]; e < &lp->e[NTYPES]; e++)
				if (e->flag == 2 && e->type == 0xffff)
					break;
			if (e == &lp->e[NTYPES]) {
				efreeb(bp);
				continue;
			}
		}
		bp->wptr += lp->rring[lp->rl].cntflags - 4;
		if (e->q->flag & QFULL) {
			efreeb(bp);
			continue;
		}
		do {
			bp2 = allocb(bp->wptr - bp->rptr);
			size = min(bp->wptr - bp->rptr, bp2->lim - bp2->base);
			bcopy(bp->rptr, bp2->wptr, size);
			bp->rptr += size;
			bp2->wptr += size;
			if (bp->rptr == bp->wptr)
				bp2->class |= S_DELIM;
			if (bp2->class == 076)
				printf("cm:upsrv: cmblock escaped 2\n");
			(*e->q->qinfo->putp)(e->q, bp2);
		} while (bp->rptr < bp->wptr);
		efreeb(bp);
	}

	/*
	 *  look for a xmitt'd packets, wake any process waiting for a
	 *  transmit buffer.
	 */

	xrfreed = 0;
	while (lp->tl != lp->tc && (lp->tring[lp->tl].flags & OWN) == 0) {
		if (lp->tring[lp->tl].flags & ERR) {
			printf("cm%d: ", lanceno);
			printf("xmit error 0x%x 0x%x\n",
				lp->tring[lp->tl].flags,
				lp->tring[lp->tl].cntflags);
		}
		efreeb(lp->tbp[lp->tl]);	/* put block back */
		lp->tl = TSUCC(lp->tl);
		xrfreed++;
	}
	if (xrfreed)
		cmwakeq(lanceno);	/* see if any queue wants to write */
	stagerbuf(lanceno);
	if (cmdebug & DBREST)
		printf("cmint bot: csr 0x%x\n", *lp->rdp);
}


static
cminit()
{
	struct block  *bp;
	int i;
	Initblock *li;
	register a;
	register struct lanceram *lrp;
	Msg *msg;

	if (cmdebug & DBREST)
		printf("cminit: top \n");

	/* Set up pointers to ncm's ram */

	lrp = (struct lanceram *)LANCERAM;
	bzero((char *)lrp, sizeof (struct lanceram));

	/* initialize the free list */

	for (i = 0; i < NRBLOCKS; i++) {
		bp = lb.blocks + i;
		bp->base = (u_char *)&lrp->e[i];
		bp->lim = bp->base + sizeof(Etherp);
		bp->class = 076;	/* to keep them separate */
		bp->next = lb.free;
		lb.free = bp;
	}
}

static
linit(lnum)
int lnum;
{
	int i;
	Initblock *li;
	register struct lanceram *lrp;
	Lance *lp;

	if (cmdebug & DBREST)
		printf("linit: top for lance %x\n", lnum);

	/* Set up pointers to ncm ram and Lance structure */

	lrp = (struct lanceram *)LANCERAM;
	li = &lrp->initblock[lnum];
	lp = &l[lnum];


	lp->rring = &lrp->rring[lnum][0];
	lp->tring = &lrp->tring[lnum][0];
	lp->rap=lnum?(u_short *)((u_int)LANCEAP+0x100000):(u_short *)LANCEAP;
	lp->rdp=lnum?(u_short *)((u_int)LANCEDP+0x100000):(u_short *)LANCEDP;
	*lp->rap = 0;
	*lp->rdp = STOP;		/* in case it was running */

	/* point lance to the initialization block */

	*lp->rap = 1;
	*lp->rdp = LADDR(li);
	*lp->rap = 2;
	*lp->rdp = HADDR(li);

	/* The lance byte swaps until we tell it not to */

	*lp->rap = 3;
	*lp->rdp = BSWP;

	/* creat the initialization block */

	li->mode = 0;

	/* set ether address from battery backed up ram */

	li->etheraddr[0] = 0x2200;
	li->etheraddr[1] = 0;
	li->etheraddr[2] = (u_short)0x8000 + complex - 1;
	bcopy((caddr_t)li->etheraddr, lp->ea, 6);
	swab((caddr_t)li->etheraddr, (caddr_t)li->etheraddr, 6);

	/* set multicast hash bit */
	for (i = 0; i < 4; i++)
		li->multi[i] = 0;
	li->multi[0] = 2;
	li->multi[1] = 0x1000;
	if (cmdebug & DBREST)
		printf("linit: bot for lance %x\n", lnum);

}

static
binit()
{
	Lance *lp;
	int i;

	if (cmdebug & DBREST)
		printf("binit: top\n");

	for (i=0;i<=1;i++) {
		lp = &l[i];
		*lp->rap = 0;
		stagerbuf(i);
		stagetbuf(i);
		*lp->rap = 0; /* init lance, turn on int, turn on xmt,rcv */
		if (cmdebug & DBREST)
			printf("binit: pre-init csr 0x%x\n", *lp->rdp);
		*lp->rdp = INEA|INIT;
		{ int i; for (i=0; i<10000; i++); }
		if (cmdebug & DBREST)
			printf("binit: bot csr 0x%x\n", *lp->rdp);
	}
}


/*  stage as many receive buffers as possible */

static
stagerbuf(lnum)
int lnum;
{
	int next;
	struct block  *bp;
	Lance *lp;
	int i = 0;
	Initblock *li;
	struct lanceram *lrp;

	if (lnum < 0 || lnum > 1)
		panic("stagerbuf: lnum %d\n", lnum);
	lrp = (struct lanceram *)LANCERAM;
	lp = &l[lnum];
	li = &lrp->initblock[lnum];

	for (next = RSUCC(lp->rc); next != lp->rl; next = RSUCC(next)) {
		lp->rbp[lp->rc] = bp = eallocb();
		if (bp == 0) {
			printf("no lance buffers!\n");
			break;
		}
		lp->rring[lp->rc].size = -(bp->lim - bp->base);
		lp->rring[lp->rc].cntflags = 0;
		lp->rring[lp->rc].laddr = LADDR(bp->base);
		lp->rring[lp->rc].flags = OWN|HADDR(bp->base);
		lp->rc = next;
		i++;
	}
	if (cmdebug & DBREST)
		printf("staged %d buffers-csr 0x%x\n", i, *lp->rdp);
	li->rdralow = LADDR(lp->rring);
	li->rdrahigh = (LOGNRRB<<13) | HADDR(lp->rring);

}

static
stagetbuf(lnum)
int lnum;
{
	Lance *lp;
	Initblock *li;
	struct lanceram *lrp;
	int i;

	if (cmdebug & DBREST)
		printf("stagetbuf: top for lance %d\n",lnum);

	lp = &l[lnum];
	lrp = (struct lanceram *)LANCERAM;
	li = &lrp->initblock[lnum];

	for (i = 0; i < NTRB; i++) {
		lp->tring[i].cntflags = 0;
		lp->tring[i].laddr = 0;
		lp->tring[i].flags = 0;
		lp->tring[i].size = 0;
	}

	li->tdralow = LADDR(lp->tring);
	li->tdrahigh = (LOGNTRB<<13)|HADDR(lp->tring);
	if (cmdebug & DBREST)
		printf("stagetbuf: bot for lance %d\n",lnum);

}

static
struct block  *
eallocb()
{
	register struct block  *bp;
	struct block *tp;
	register s;
	int i = 0;

	s = splimp();
	if ((bp = lb.free) == NULL)
		goto out;
	lb.free = bp->next;
	bp->type = M_DATA;
	bp->next = NULL;
	bp->rptr = bp->wptr = bp->base;
out:
	if (cmdebug & DBBLOCK) {
		tp = lb.free;
		while (tp->next) {
			tp = tp->next;
			i++;
		}
		printf("eallocb: free blks = %d\n", i);
	}
	if (cmdebug & DBBLOCK)
		printf("eallocb: 0x%x\n", bp);
	splx(s);
	return bp;
}


static
efreeb(bp)		/* put a block back on the free list */
	register struct block  *bp;
{
	register s;
	struct block *tp;
	int i = 0;

	s = splimp();
	if (bp->class != 076)
		printf("efreeb: this is not my block\n");
	bp->next = lb.free;
	lb.free = bp;
	if (cmdebug & DBBLOCK) {
		tp = lb.free;
		while(tp->next) {
			tp = tp->next;
			i++;
		}
		printf("efreeb: free blks = %d\n", i);
	}
	if (cmdebug & DBBLOCK)
		printf("efreeb: 0x%x\n", bp);
	splx(s);
}

static
printpacket(p, lp)
	Etherp *p;
	Lance *lp;
{
	printf("%d 0x%x\n", inpackets, *lp->rdp);
	printf("d(%x.%x.%x.%x.%x.%x)\n",
		p->d[0], p->d[1], p->d[2], p->d[3], p->d[4], p->d[5]);
	printf("s(%x.%x.%x.%x.%x.%x)\n",
		p->s[0],  p->s[1],  p->s[2],  p->s[3],  p->s[4],  p->s[5]);
	printf("t(%x)\n", p->type);
}

cmwakeq(lance)		/* check for queue waiting for space in transmit ring*/
int lance;
{
	register Ethertype *e;

	for (e = l[lance].e; e < &l[lance].e[NTYPES]; e++)
		if (e->flag)
		if (e->q->count)
			qenable(e->q);
}

static
cmprbp(s, bp)		/* print things about blocks */
	char *s;
	register struct block *bp;
{
	printf("%s:base 0x%x, lim 0x%x\n", s, bp->base, bp->lim);
	printf("%s:rptr 0x%x, wptr 0x%x, count %d\n", s, bp->rptr, bp->wptr,
		bp->wptr - bp->rptr);
}
cmdopen(dev, flag)
	dev_t dev;
{
	if (fushort(LANCERAM) == -1) {
		u.u_error = ENXIO;
		return;
	}
}

short fdata[7];
char	*sputnp;

cmdread(dev)
{
	register char *p;
	u_short *fp;
	int n;
	u_short sh;
	char buf[6];
	char *abp, *cpfp;
	static char ab[] = {'n','o','a','u','t','o','b','o','o','t','\n','\0'};
	static char cpf[]={'n','o','c','o','n','p','r','i','n','t','\n','\0'};

	abp = &ab[2];
	cpfp = &cpf[2];
	fp = FIFO;
	*fp = 0x00;
	*fp = 0x10;
	*fp = 0x55;
	if (u.u_offset > 3)
		return;
	sleep(FIFO, CMPRI);
	switch(minor(dev)) {
	case 0:
		sputnp = buf;
		for (p = buf; p < &buf[5]; p++)
			*p = ' ';
		*p = '\n';
		sputn((~fdata[2] & 0x3f)<<8 | (~fdata[3] & 0xff));
		if ((n = min(u.u_count, sizeof buf - u.u_offset)) > 0)
			iomove(buf + u.u_offset, n, B_READ);
		break;
	case 1:
		if (fdata[2] & 0x0080) {
			if ((n = min(u.u_count, strlen(ab) - u.u_offset)) > 0)
				iomove(ab + u.u_offset, n, B_READ);
		} else {
			if ((n = min(u.u_count, strlen(abp) - u.u_offset)) > 0)
				iomove(abp + u.u_offset, n, B_READ);
		}
		break;
	case 2:
		if (fdata[2] & 0x0040) {
			if ((n = min(u.u_count, strlen(cpf) - u.u_offset)) > 0)
				iomove(cpf + u.u_offset, n, B_READ);
		} else {
			if ((n = min(u.u_count, strlen(cpfp)-u.u_offset)) > 0)
				iomove(cpfp + u.u_offset, n, B_READ);
		}
		break;
	}
}
cmfifointr()
{
	u_short *fp;
	int i;
	fp = FIFO;
	for (i=0;i<7;i++)
		fdata[i] = *fp;
	wakeup(FIFO);
}

sputn(n)
	register n;
{
	if (n >= 10)
		sputn(n/10);
	*sputnp++ = (n%10) + '0';
}
strlen(s)
register char *s;
{
	register char *s0 = s + 1;

	while (*s++ != '\0')
		;
	return (s - s0);
}

memcmp(s1, s2, n)
register char *s1, *s2;
register n;
{
	while (n--)
		if (*s1 != *s2)
			return *s1 - *s2;
		else
			s1++, s2++;
	return 0;
}
