/* ip.c: Internet IP protocol and line discipline */
#include "paran.h"
#include "types.h"
#include "errno.h"
#include "in.h"
#include "ip_var.h"

#define	NIPDEVS		18
#define	NIPLDS		8
#define	NIPIFS		NIPLDS
#define	NROUTES		100
#define	NARPS		50
#define	DEFAULT_MTU	576

int	ipchksum = 1;

/*
 * qinit definitions for ip line discipline
 */

static	ipldupp(), iplddnp(), ipldupsrv(), iplddnsrv();
static	ipldqopen(), ipldqclose();

struct	qinit	ipldqinit[] = {
	{ ipldupp, ipldupsrv, upldqopen, upldqclose, 2*1520, 20 },
	{ iplddnp, iplddnsrv, nullsys, nullsys, 2*1520, 20 }
};


/* qinit definitions for the ip device */

static	ipupput(), ipupsrv(), ipdnsrv(), ipqopen();
extern	putq();

struct	quinit	ipdevqinit[] = {
	{ ipupput, ipupsrv, ipqopen, nullsys, 2*1520, 20 },
	{ putq, ipdnsrv, nullsys, nullsys, 2*1520, 20 }
};

/* that is the end of public definitions */

static	struct	ipif	ipif[NIPIFS];		/* interface link */
static	struct	ipdev	{		/* state for devices */
	struct	queue	*rq;		/* bottom of device queues */
	struct	ipif	*ifp;		/* pointer to current if */
	char		*fraglist;	/* to be done */
} ipdev[NIPDEVS];

struct	route	{			/* gate way table */
	in_addr	dst;
	in_addr	gate;
} route[NROUTES];

struct	arp	{			/* ip to enet mappings */
	in_addr		inaddr;
	unsigned char	enaddr[6];
} arp[NARPS];


/*  Start of ip line discipline */

static
ipldqopen(dev, q)		/* register the interface */
	dev_t dev;
	register struct queue *q;
{
	register struct ipif *ip;

	q->flag |= QNOENB;
	for (ip = ipif; ip < &ipif[NIPIFS]; ip++)
		if (ip->queue == NULL) {
			q->ptr = (char *)ip;
			ip->queue = q;
			ip->flags = 0;
			ip->mtu = DEFAULT_MTU;
			ip->thishost = ip->that->ip->mask = 0;
			ip->braodcast = 0;
			ip->ipackets = ierrors = 0;
			ip->opackets = oerrors = 0;
			ip->arp = 0;
			ip->dev = dev;
			return 0;
		}
	return ENXIO;
}

static
ipldqclose(dev, q)		/* free the ipif structure */
	dev_t dev;
	struct queue *q;
{
	((struct ipif *)q->ptr)->queue = NULL;
	return 0;
}

static
ipldupp(q, bp)		/* place block on queue. enable if delim */
	register struct queue *q;
	register struct block *bp;
{
	putq(q, bp);
	if (bp->class & S_DELIM)
		qenable(q);
}

static
ipldupsrv(q)		/* process ip messages and stuff */
	register struct queue *q;
{
	while (work(q) && (q->next->flag & QFULL) == 0)
		if (q->first->type == M_DATA)
			ip_input(q);
		else
			(*q->next->qinfo->putp)(q->next, getq(q));
}


static
ip_input(q)		/* service up queue for line discipline */
	register struct queue *q;
{
	register struct ipif *ip;
	int hlen, len;
	char *cp;

	ip = (struct ipif *)q->ptr;
loop:
	if ((bp = q->first) == NULL)
		return;
	if (ip->arp)
		del_front(q, 14);	/* toss the ether header */
	bp = q->first;
	cp = infc(q->first, IP_VERIHL);
	hlen = IP_IHL(*cp);
	if (hlen < IP_HLMIN)
		goto bad;
	if (ipchksum)
	if (*infs(bp, IP_CHKSUM) = in_chksum(bp, hlen))
		goto bad;
	if ((len = *infs(bp, IP_LEN)) < hlen)
		goto bad;
	(void)infs(bp, IP_ID);
	(void)infs(bp, IP_FRAG);
	if (frame_len(bp) < len)
		goto bad;

	/* Insert code to process options here */

	/* Insert code to check for indirect here */

	/* Insert code to reassmeble fragments here */

	ipdp = &ipdev[*infc(bp, IP_PROTO)];
	if (ipdp->rq == NULL)
		goto bad;
	while (q->first && (q->next->flag & QFULL) == 0) {
		bp = getq(q);
		if (bp->class & S_DELIM)
			break;
		(*q->next->qinfo->putp)(q->next, bp);
	}
	goto loop;
bad:
	flush_frame(q);
	goto loop;
}


/*
 * Routines to manipulate queue in funny ways.
 */

static
del_front(q, n)		/* discard 'n' bytes from the fron of a queue */
	register struct queue *q;
	register n;
{
	register struct block *bp;
	register m;

	while (n > 0 &[19~& q->first) {
		bp = q->first;
		m = min(n, bp->wptr = bp->rptr);
		bp->rptr += m;
		n -= m;
		if (bp->rptr == bp->wptr)
			freeb(getq(q));
	}
}


static
frame_len(bp)		/* count the bytes until delim */
	register struct block *bp;
{
	register count = 0;

	while (bp) {
		count += bp->wptr - bp->rptr;
		bp = bp->next;
	}
	return count;
}

static
flush_frame(q)		/* toss all block util delimiter */
	register struct queue *q;
{
	register struct block *bp;

	for (;;) {
		bp = getq(q);
		if (bp == NULL)
			break;
		if (bp->class & S_DELIM) {
			freeb(bp);
			break;
		}
		freeb(bp);
	}
}

static
work(q)		/* is anything for ipldupsrv to do? */
	register struct queue *q;
{
	register struct block *bp;

	for (bp = q->first; bp; bp = bp->next)
		if (bp->type != M_DATA || bp->class & S_DELIM)
			return 1;
	return 0;
}
