/* stream.c:  Dennis Ritchie streams.  Yet another version */
#include "param.h"
#include "dir.h"
#include "signal.h"
#include "proc.h"
#include "inode.h"
#include "file.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "user.h"
#include "filio.h"
#include "ttyio.h"
#include "stream.h"
#include "errno.h"
#include "buf.h"

#define NULL 0
#define	BLBUFSIZE	4096
#define	STIPRI		28
#define	STOPRI		29

struct	strunq	{
	struct	queue	*first;
	struct	queue	*last;
} strunq;
static	struct	class	class[6];
static	struct	stdata	*stlist;
static	struct	stdata	stdata[NSTREAMS];
static	struct	queue	queue[NQUEUES];
static	struct	block	block[NBLOCKS];
static	u_char	blkbuf[NBLBUFS * BLBUFSIZE];	/* where blocks come from */
static	stdebug = 0;
extern	int nulldev();
extern	int lbolt;		/* clock does a wakup on this every second */
extern	int selwait;
int	putq(), srv(), kdnsrv(), kupsrv();
int	kupputq();
int	kupclose(), kdnclose();

struct	qinit kqinit[] = {
/* up   */ { kupputq, kupsrv, nulldev, kupclose, 1024, 32 },
/* dn   */ { putq, kdnsrv, nulldev, nulldev, 1024, 32 }
};


stinit()			/* setup stuff for steams */
{
	register struct stdata *sp;

	for (sp = stdata; sp < &stdata[NSTREAMS]; sp++) {
		sp->wrq = (struct queue *)stlist;
		stlist = sp;
	}
	initb();		/* set up blocks */
}


stopen(dev, ip, dqinit)		/* open a stream */
	dev_t dev;
	struct inode *ip;
	struct	qinit *dqinit;
{
	register struct queue *kq, *dq;
	struct queue *qalloc();
	register struct stdata *sp;
	struct stdata *stalloc();

	plock(ip);
	kq = qalloc();
	dq = qalloc();
	qopen(kq, kqinit);
	qopen(dq, dqinit);
	dq[0].next = &kq[0];
	kq[1].next = &dq[1];
	sp = stalloc();
	kq->ptr = (char *)sp;
	OTHERQ(kq)->ptr = (char *)sp;
	sp->wrq = OTHERQ(kq);
	sp->flag = STWOPEN;
	sp->inode = ip;
	ip->i_stream = sp;
	(*dq[0].qinfo->qopen)(dev, &dq[0]);
	/* above might have freed the queues */
	if (dq[1].flag & QUSE)
		(*dq[1].qinfo->qopen)(dev, &dq[1]);
	if (u.u_error) {
		stfree(sp);
		ip->i_stream = NULL;
		qfree(kq);
		qfree(dq);
		prele(ip);
		return;
	}
	sp->flag &= ~STWOPEN;
	prele(ip);
}

/*
 * The stack of modules is poped form the stream head.
 * The server routines are all given a change to run,
 * and thee is a timeout (30 sec) for each queue so long
 * as there is data on it.  That is, each queue is given some
 * time before its downstream data is forcibly thrown away.
 * Upstream data can of course be thrown away without regrets
 * since there is no one to read it.
 *		-dmr 11/7/88
 */

stclose(dev, ip)		/* close the stream down */
	dev_t dev;
	struct inode *ip;
{
	register struct queue *wrq, *rdq;
	register struct stdata *sp;
	register tics = 0;
	register struct queue *q;
	struct block *bp;
	register notempty;
	struct queue *nq;

	sp = ip->i_stream;
	wrq = sp->wrq;
	if ((wrq->flag & QUSE) == 0)
		panic("stclose: bad queue");
	if ((sp->flag & HUNGUP) == 0) {
		bp = allocb(0);
		bp->type = M_HANGUP;
		(*wrq->qinfo->putp)(wrq, bp);
		do {
			if (notempty = !qempty(wrq)) {
				tics++;
				sleep(&lbolt, PUSER);
			}
		} while (tics < 2 && notempty);
	}
	splimp();
	q = wrq;
	while (q && (q->flag & QREADR) == 0) {
/*		printf ("stclose: #%x\n", q);*/
		flushq(q);
		flushq(OTHERQ(q));
		(*OTHERQ(q)->qinfo->qclose)(dev, OTHERQ(q));
		(*q->qinfo->qclose)(dev, q);
		nq = q->next;
		qfree(q);
		q = nq;
	}
	if (q) {
		OTHERQ(q)->next = NULL;
		while (q->next)
			q = q->next;
		((struct stdata *)(q->ptr))->flag |= HUNGUP;
	}
	spl0();
	stfree(sp);
	ip->i_stream = NULL;
}

static
kupclose(dev, q)		/* kernel close of a pipe */
	register struct queue *q;
{
	register struct stdata *sp;

	sp = (struct stdata *) q->ptr;
	sp->flag |= HUNGUP;
	if (sp->pgrp)
		signal(sp->pgrp, SIGHUP);
	if (sp->flag & RSLEEP) {
		wakeup((int)sp+0);
		wakeup(OTHERQ(sp->wrq));
	}
	if (sp->flag & WSLEEP) {
		wakeup((int)sp+1);
		wakeup(sp->wrq);
	}
	if (sp->flag & IOCWAIT) {
		wakeup((int)sp+3);
		wakeup(sp->iocblk);
	}
}


static
stflush(q)		/* toss all blocks on stream */
	register struct queue *q;
{
	register struct queue *next;

	while (q) {
		next = q->next;
		flushq(q);
		flushq(OTHERQ(q));
		q = next;
	}
}


static
qempty(q)		/* is any data in the queue */
	register struct queue *q;
{
	register s;

	s = splimp();
	while (q) {
		if (q->first) {
			splx(s);
			return 0;
		}
		q = q->next;
	}
	splx(s);
	return 1;
}

#define	BIGB	700	/* ADAPTIVE gateway limit  */

stwrite(ip)		/* write data into a stream */
	struct inode *ip;
{
	register struct block *bp;
	struct block *allocb();
	register int size;
	register struct stdata *sp;

	sp = ip->i_stream;
	stlock(sp, WSLEEP);
	splnet();
	do {
		if (sp->flag & HUNGUP) {
			u.u_error = EPIPE;
			psignal(u.u_procp, SIGPIPE);
			break;
		}
		if (sp->wrq->flag & QFULL) {
			sleep(sp->wrq, STOPRI);
			continue;
		}
		if (sp->wrq->flag & QBIGB)
			bp = allocb(BIGB);
		else
			bp = allocb(64);
		bp->type = M_DATA;
		size = min(bp->lim - bp->base, u.u_count);
		iomove(bp->wptr, size, B_WRITE);
		bp->wptr += size;
		if (u.u_count == 0)
			bp->class |= S_DELIM;
		(*sp->wrq->qinfo->putp)(sp->wrq, bp);
	} while (u.u_count && u.u_error == 0);
	stunlock(sp, WSLEEP);
	spl0();
}


stread(ip)		/* read bytes from stream head */
	struct inode *ip;
{
	register struct block *bp;
	struct block *getq();
	register struct stdata *sp;
	register size, delim = 0;

	sp = ip->i_stream;
	stlock(sp, RSLEEP);
	splnet();
	do {
		if (sp->flag & HUNGUP) {
			if (sp->count++ >= 64) {
				u.u_error = EPIPE;
				psignal(u.u_procp, SIGPIPE);
			}
			break;
		}
		if ((bp = getq(OTHERQ(sp->wrq))) == NULL) {
			sleep(OTHERQ(sp->wrq), STIPRI);
			continue;
		}
		switch (bp->type) {
		case M_DATA:
			if((size = bp->wptr - bp->rptr) > u.u_count) {
				size = u.u_count;
				iomove(bp->rptr, size, B_READ);
				bp->rptr += size;
				putbq(OTHERQ(sp->wrq), bp);
				break;
			}
			iomove(bp->rptr, size, B_READ);
			delim = bp->class & S_DELIM;
			freeb(bp);
			break;
		case M_IOCTL:
			bp->type = M_IOCNAK;
			qreply(OTHERQ(sp->wrq), bp);
			break;
		case M_HANGUP:
			sp->flag |= HUNGUP;
			freeb(bp);
			break;
		default:
			printf("stread: bad type (0%o)\n", bp->type);
			freeb(bp);
		}
	} while (u.u_count && u.u_error == 0 && delim == 0);
	stunlock(sp, RSLEEP);
	spl0();
}

stselect(sp, mode)		/* see if we can read/write this stream */
	register struct stdata *sp;	/* from i_stream */
{
	register struct queue *wrq;

	splnet();
	wrq = sp->wrq;
	switch (mode) {
	case FREAD:
		if (OTHERQ(wrq)->count > 0)
			return 1;
		splimp();
		if (sp->rsel)
			sp->flag |= RSEL;
		sp->rsel = u.u_procp;
		break;
	case FWRITE:
		if (wrq->count < wrq->qinfo->lolimit)
			return 1;
		splimp();
		if (sp->wsel)
			sp->flag |= WSEL;
		sp->wsel = u.u_procp;
		break;
	default:
		printf("stselect: bad mode\n");
	}
	spl0();
	return 0;
}

static
selwakeup(sp, mode)		/* Hello? Hello? */
	register struct stdata *sp;
{
	register struct proc *pp;
	register mask;
	register short *fp;
	register s;

	s = splimp();
	mask = (mode == FREAD) ? RSEL : WSEL;
	fp = &sp->flag;
	pp = (mode == FREAD) ? sp->rsel : sp->wsel;
	if (pp == NULL)
		return;
	if (pp->p_flag & SSEL)
		pp->p_flag &= ~SSEL;
	if (*fp & mask) {
		wakeup(&selwait);
		*fp &= ~mask;
	} else
		wake1up(&selwait, pp);
	splx(s);
}

extern	int	nlds;
extern	struct	qinit *qinit_ld[];


stioctl(sp, cmd, addr, dev)		/* io control */
	register struct stdata *sp;
	caddr_t addr;
{
	register struct block *bp;
	register struct stioctl *stp;
	register now, n, ld;
	register struct queue *ldq;
	short pgrp = 0;
	int ioctlto();
	int c;

	while (sp->flag & IOCWAIT)
		sleep((int)sp+1, STIPRI);
	sp->flag |= IOCWAIT;
	switch (cmd) {
	case FIONREAD:
		c = countq(OTHERQ(sp->wrq));
		if (copyout((char *)&c, addr, sizeof (int)) < 0)
			u.u_error = EFAULT;
		break;
	case FIOPUSHLD:
		if ((ld = fuword(addr)) == -1) {
			u.u_error = EFAULT;
			break;
		}
		if (ld < 0 || ld >= nlds || qinit_ld[ld] == NULL) {
			u.u_error = EINVAL;
			break;
		}
		splimp();
		ldq = qalloc();
		qopen(ldq, qinit_ld[ld]);
		ldq[1].next = sp->wrq->next;
		sp->wrq->next = &ldq[1];
		ldq[0].next = OTHERQ(ldq[1].next)->next;
		OTHERQ(ldq[1].next)->next = &ldq[0];
		u.u_error = (*ldq[0].qinfo->qopen)(dev, &ldq[0]);
		if (u.u_error == 0)
			u.u_error = (*ldq[1].qinfo->qopen)(dev, &ldq[1]);
		spl0();
		if (u.u_error) {
			sp->wrq->next = ldq[1].next;
			OTHERQ(sp->wrq->next)->next = ldq[0].next;
			qfree(ldq);
			break;
		}
		break;
	case FIOPOPLD:
		c = 0;
		if (addr != NULL && (c = fuword(addr)) == -1) {
			u.u_error = EFAULT;
			break;
		}
		popq(sp->wrq, c, dev);	/* pop the top queue off */
		break;
	case FIOLOOKLD:
	case FIOINSLD:
	case FIOSNDFD:
	case FIORCVFD:
		u.u_error = EIO;
		break;
	case TIOCSPGRP:	/* set process group */
		if (addr && (pgrp = fushort(addr)) == -1) {
			u.u_error = EFAULT;
			break;
		}
		if (pgrp == 0)
			pgrp = u.u_procp->p_pid;
		u.u_procp->p_pgrp = pgrp;
		u.u_ttyp = NULL;
		u.u_ttyd = dev;
		sp->pgrp = pgrp;
		break;
	case TIOCGPGRP:	/* get process group */
		if (sushort(addr, sp->pgrp) == -1)
			u.u_error = EFAULT;
		break;
	default:	/* must be for the device */
		bp = allocb(sizeof (struct stioctl));
		bp->type = M_IOCTL;
		bp->wptr += STIOCSIZE + STIOCHDR;
		stp = (struct stioctl *) bp->base;
		stp->com[0] = cmd;
		stp->com[1] = cmd >> 8;
		if (copyin(addr, (caddr_t) &stp->data[0], STIOCSIZE) < 0) {
			u.u_error = EFAULT;
			freeb(bp);
			break;
		}
		sp->iocblk = NULL;
		timeout(ioctlto, &sp->iocblk, 10*HZ);
		now = lbolt;
		(*sp->wrq->qinfo->putp)(sp->wrq, bp);
		splimp();
		while (sp->iocblk == NULL && lbolt < now+(10*HZ))
			sleep(&sp->iocblk, STIPRI);
		if (now+(10*HZ) <= lbolt) {
			untimeout(ioctlto, &sp->iocblk);
			u.u_error = ENXIO;
			spl0();
			break;
		}
		spl0();
		bp = sp->iocblk;
		untimeout(ioctlto, &sp->iocblk);
		if (bp->type == M_IOCNAK) {
			u.u_error = ENXIO;
			freeb(bp);
			break;
		}
		if ((n = bp->wptr - bp->rptr) > 0)
			if (copyout((caddr_t)bp->rptr, addr, n) < 0)
				u.u_error = EFAULT;
		freeb(bp);
	}
	sp->flag &= ~IOCWAIT;
	wakeup((int)sp+2);
}

static
popq(q, n, dev)		/* pop line discipline 'n' from queue 'q' */
	register struct queue *q;
	int n;
{
	register struct queue *ldq;

	ldq = q->next;
	while (n) {
		if (ldq->next == NULL)
			break;
		if (ldq->next->flag & QREADR)
			break;
		ldq = ldq->next;
		n--;
	}
	if (n != 0 || ldq->next == NULL) {
		u.u_error = EINVAL;
		return;
	}
	splimp();
	flushq(ldq);
	flushq(OTHERQ(ldq));
	(*OTHERQ(ldq)->qinfo->qclose)(dev, OTHERQ(ldq));
	(*ldq->qinfo->qclose)(dev, ldq);
	OTHERQ(OTHERQ(ldq)->next)->next = ldq->next;
	OTHERQ(ldq->next)->next = OTHERQ(ldq)->next;
	qfree(ldq);
	spl0();
}


static
ioctlto(wchan)	/* ioctl timeout */
	caddr_t wchan;
{
	if (stdebug)
		printf("ioctlto: wchan 0x%x\n", wchan);
	wakeup(wchan);
}


static
countq(q)		/* how many bytes this end of the queue */
	register struct queue *q;
{
	register s;
	register cc = 0;
	register struct block *bp;

	s = splimp();
	for (bp = q->first; bp; bp = bp->next)
		if (bp->type == M_DATA)
			cc += bp->wptr - bp->rptr;
	splx(s);
	return cc;
}

struct queue *
qalloc()		/* return pointer to first of a pair of queue */
{
	register struct queue *qp;

	for (qp = queue; qp < &queue[NQUEUES]; qp += 2)
		if (!(qp->flag & QUSE)) {
			qp[0].flag |= QUSE|QREADR;
			qp[1].flag |= QUSE;
			return qp;
		}
	panic("qalloc: out of queues!");
}


/* doesn't matter which queue is given to qfree */

qfree(q)		/* free a pair of queue */
	register struct queue *q;
{
	register int i;
	register struct queue *next;

	for (i = 0; i < 2; i++) {
		next = OTHERQ(q);
		if (q->flag & QENAB)		/* remove from enable q */
			qdisable(q);
		if ((q->flag & QUSE) == 0)
			panic("qfree: freed free queue");
		q->flag = 0;
		q->qinfo = NULL;
		q->next = NULL;
		q->link = NULL;
		q->first = q->last = NULL;
		q->count = 0;
		q->ptr = NULL;
		q = next;
	}
}

static
qopen(q, qinit)		/* initailize a pair of queues */
	struct queue *q;
	struct qinit qinit[];
{
	q[0].qinfo = &qinit[0];
	q[1].qinfo = &qinit[1];
	q[0].count = q[1].count = 0;
	q[0].first = q[0].last = q[1].first = q[1].last = NULL;
}

qenable(q)
	register struct queue *q;
{
	extern struct queue *enabled;
	register s;

	if (q == 0)
		return;
	s = splimp();
	if (q->flag & QENAB) {
		splx(s);
		return;
	}
	q->link = NULL;
	if (strunq.first == NULL)
		strunq.first = q;
	else
		strunq.last->link = q;
	strunq.last = q;
	q->flag |= QENAB;
	setqsched();
	splx(s);
}

qdisable(q)	/* take off runq if enabled */
	register struct queue *q;
{
	register struct queue *p;
	register s;

	s = splimp();
	if ((p = strunq.first) == NULL) {
		splx(s);
		return;
	}
	if (p == q) {
		if ((strunq.first = q->link) == NULL)
			strunq.last = NULL;
	} else {
		while (p->link && p->link != q)
			p = p->link;
		if (p->link == NULL) {
			splx(s);
			return;
		}
		/* here means p->link == q */
		if ((p->link = q->link) == NULL)
			strunq.last = p;
	}
	splx(s);
}

stintr()		/* soft interrupt routine to run srv routines */
{
	register struct queue *q;
	register int runable;
	register s;

	pcc_reset_si1();		/* turn off soft intr */
	while (strunq.first != NULL) {
		s = splimp();
		q = strunq.first;
		if ((strunq.first = strunq.first->link) == NULL)
			strunq.last = NULL;
		runable = (q->flag & QENAB);
		q->flag &= ~QENAB;
		splx(s);
		if (runable && q && q->qinfo && q->qinfo->srvp)
			(*q->qinfo->srvp)(q);
	}
}

srv(q)		/* generic service procedure */
	register struct	queue *q;
{
	struct block *getq();
	register struct block *bp;

	if (q->next == NULL)
		return;
	while ((bp = q->first) && (bp->type&QPCTL || (q->next->flag&QFULL) == 0))
		(*q->next->qinfo->putp)(q->next, getq(q));
	if (q->first)
		q->next->flag |= QWANTW;
}

static
kupputq(q, bp)		/* queue stuff on kernel stream head */
	register struct queue *q;
	register struct block *bp;
{
	register struct stdata *sp;

	if (stdebug)
		printf("kuppputq: bp->type = 0%o\n", bp->type);
	sp = (struct stdata *)q->ptr;
	if (bp->type == M_IOCNAK || bp->type == M_IOCACK)
		if (sp->iocblk == NULL) {
			sp->iocblk = bp;
			wakeup(&sp->iocblk);
		} else
			freeb(bp);	/* nobody cares */
	else if (bp->type == M_SIGNAL) {
		signal(sp->pgrp, (u_char)*bp->rptr);
		freeb(bp);
	} else
		putq(q, bp);
}



static
kupsrv(q)	/* kernel upstream service procedure */
	register struct queue *q;
{
	if (stdebug)
		printf("kupsrv: 0x%x\n", q);
	wakeup(q);
	selwakeup((struct stdata *)q->ptr, FREAD);
	if (stdebug)
		printf("kupsrv: bot\n");
}

static
kdnsrv(q)		/* kernel downstream service procedure */
	register struct queue *q;
{
	if (stdebug)
		printf("kdnsrv: top 0x%x\n", q);
	srv(q);
	if (q->count < q->qinfo->lolimit) {
		wakeup(q);
		selwakeup((struct stdata *)q->ptr, FWRITE);
	}
	if (stdebug)
		printf("kdnsrv: bot=\n");
}

/*
 * putq, putbq, getq, qsize, flushq all manipulate the blocks on a queue.
 */


putq(q, bp)		/* put block onto queue */
	register struct queue *q;
	register struct block *bp;
{
	register s;

	if (bp->next != NULL)
		panic("putq: non-null bp->next pointer");
	s = splimp();
	if (stdebug)
		printf("putq(0x%x, 0x%x)\n", q, bp);
	if (q->first == NULL && (q->flag & QNOENB) == 0)
		qenable(q);
	if (bp->type & QPCTL) {
		putbq(q, bp);
		if (stdebug)
			printf("putq: bot\n");
		splx(s);
		return;
	} else if (q->first == NULL)
		q->first = bp;
	else
		q->last->next = bp;
	q->last = bp;
	if (q->flag&QWANTR && q->next)
		qenable(q->next);
	q->count += bp->lim - bp->base;
	if (q->count >= q->qinfo->limit)
		q->flag |= QFULL;
	if (stdebug)
		printf("putq: bot\n");
	splx(s);
}



putbq(q, bp)	/* return block 'bp' to front of queue */
	register struct queue *q;
	register struct block *bp;
{
	register s;

	s = splimp();
	if (stdebug)
		printf("putbq: top q 0x%x, bp 0x%x\n", q, bp);
	bp->next = q->first;
	q->first = bp;
	if (q->last == NULL)
		q->last = bp;
	q->count += bp->lim - bp->base;
	if (q->count >= q->qinfo->limit)
		q->flag |= QFULL;
	if (stdebug)
		printf("putbq: bot\n");
	splx(s);
}

qreply(q, bp)		/* send block up the stream */
	register struct queue *q;
	register struct block *bp;
{
	if (stdebug)
		printf("qreply: top.  q 0x%x, bp 0x%x\n", q, bp);
	(*OTHERQ(q)->qinfo->putp)(OTHERQ(q), bp);
	if (stdebug)
		printf("qreply: bot\n");
}

struct block *
getq(q)		/* get a message from a queue */
	register struct queue *q;
{
	register struct block *bp;
	register s;

	s = splimp();
	q->flag &= ~QWANTR;
	if ((bp = q->first) == NULL) {
		splx(s);
		return NULL;
	}
	if ((q->first = bp->next) == NULL)
		q->last = NULL;
	q->count -= bp->lim - bp->base;
	if (q->flag & QFULL && q->count < q->qinfo->lolimit) {
		q->flag &= ~QFULL;
		if (q->flag & QWANTW)  {
			q->flag &= ~QWANTW;
			qenable(OTHERQ(OTHERQ(q)->next));
		}
	}
	splx(s);
	bp->next = NULL;
	return bp;
}

flushq(q)		/* free all blocks on the queue */
	struct queue *q;
{
	register struct block *bp;
	register s;

	s = splimp();
	while (bp = getq(q))
		freeb(bp);
	splx(s);
}


struct stdata *
stalloc()
{
	register struct stdata *ssp;

	if ((ssp = stlist) == NULL) 
		panic("out of stdata");
	stlist = (struct stdata *)ssp->wrq;
	ssp->wrq = NULL;
	ssp->inode = NULL;
	ssp->flag = 0;
	return ssp;
}


stfree(ssp)
	register struct stdata *ssp;
{
	ssp->wrq = (struct queue *)stlist;
	stlist = (struct stdata *)ssp;
}



static
initb()		/* build an inventory of blocks */
{
	register struct block *bp;
	register u_char *cp;
	register i;

	cp = blkbuf;
	bp = block;
	for (i = 0; i < NBLBUFS; i++) {
		bp->base = cp;
		bp->lim = cp + BLBUFSIZE;
		cp += BLBUFSIZE;
		bp->class = 5;	/* 4k class */
		freeb(bp);
		bp++;
	}
	for (; bp < &block[NBLOCKS]; bp++) {
		bp->base = bp->data;
		bp->lim = &bp->data[4];
		bp->class = 0;
		freeb(bp);
	}
}


/* 
 * Dennis says 10th Edition allocb can return blocks smaller
 * than asked.  For the time being I'm sticking to the strick
 * version so that the inet stuff is eaiser.  Code in that part
 * of the system will assume that the blocks contain the entire
 * frame.
 */

struct	block *
allocb(size)			/* get a block from the free list */
	int	size;		/* minimum size block to allocate */
{
	register struct class *bcp;
	register struct block *bp, *bp1;
	register cl, s, i;
	register u_char *newbase;

	cl = (size > 1024)	? 5 :
		(size > 256) 	? 4 :
		(size > 64)  	? 3 :
		(size > 16)	? 2 :
		(size > 4)   	? 1 :
				  0 ;
	s = splimp();
	if ((bcp = &class[cl])->nblks) {
		bp = bcp->free;
		bcp->free = bcp->free->next;
		bcp->nblks--;
	} else if (cl == 0 || cl == 5) {
		printf("allocb: size %d\n", size);
		panic("stream blocks exhausted");
	} else {		/* get a larger one and break it up */
		bp = allocb(size << 2);
		size = (bp->lim - bp->base) / 4;
		bp->lim = bp->base + size;
		bp->class--;
		newbase = bp->lim;
		for (i = 0; i < 3; i++) {
			bp1 = allocb(4);
			bp1->class = bp->class;
			bp1->base = newbase;
			newbase += size;
			bp1->lim = bp1->base + size;
			bp1->next = NULL;
			freeb(bp1);
		}
	}
	bp->type = M_DATA;
	bp->rptr = bp->wptr = bp->base;
	bp->next = NULL;
	splx(s);
	return bp;
}

struct block *
dupb(bp)	/* return block that points to data in 'bp' */
	register struct block *bp;
{
	register struct block *bp2;

	/* assume that allocb will return 4byte block */
	bp2 = allocb(0);
	bp2->base = bp->base;
	bp2->lim = bp->lim;
	bp2->rptr = bp->rptr;
	bp2->wptr = bp->wptr;
	return bp2;
}

freeb(bp)		/* replace block on its list */
	struct block *bp;
{
	register struct class *bcp;
	register s;

	if (bp->next != NULL)
		panic("freeb: non-null next");
	s = splimp();
	bp->class &= CL_MASK;
	bcp = &class[bp->class];
	bp->next = bcp->free;
	bcp->free = bp;
	if (bp->class < 0 || bp->class > 5) {
		printf("freeb: badclass 0%o\n", bp->class);
		panic("freeb");
	}
	if (++bcp->nblks > bcp->max)
		bcp->max = bcp->nblks;
	/* coalesce blocks code would go here */
	if (bp->class == 0) {
		bp->base = &bp->data[0];
		bp->lim = &bp->data[4];
	}
	splx(s);
}

static
stlock(sp, flag)
	register struct stdata *sp;
	register flag;
{
	while (sp->flag & flag)
		sleep((int)sp+(flag>>2), STIPRI);
	sp->flag |= flag;
	u.u_stream = sp;
	u.u_stflag |= flag;
}


static
stunlock(sp, flag)
	register struct stdata *sp;
	register flag;
{
	sp->flag &= ~flag;
	u.u_stflag &= ~flag;
	u.u_stream = NULL;
	wakeup((int)sp+(flag>>2));
}


stclrlck()		/* cleanup a locked stream */
{
	stunlock(u.u_stream, u.u_stflag);
}
