/* trap.c:  Code that handles entry into the kernel. */

#include "param.h"
#include "systm.h"
#include "dir.h"
#include "signal.h"
#include "seg.h"
#include "ipm.h"
#include "sid.h"
#include "user.h"
#include "errno.h"
#include "proc.h"
#include "reg.h"
#include "psl.h"
#include "trap.h"

#define	NSYSENT	64
#define	USER	512	/* user-mode flag added to type */
struct kind {
	unsigned short ps;	/* status register */
	unsigned short voffset;	/* stack type, vector offset */
};

extern	char *sysnames[];		/* ascii names for debugging */
/*
 * Called from the trap handler when a processor trap occurs.
 */
trap(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,usp,pc,ps_kind, xxx)
	struct kind ps_kind;
{
	register i, *ar0, type;
	time_t syst;
	int adr;	 /* for BUSERR & ADDRERR */

	type = (ps_kind.voffset >> 2) & 0xff;	/* get trap type */
	if (type >= TRAP5 && type <= TRAP15)
		type = BPTFLT;
	syst = u.u_stime;
	u.u_ar0 = ar0 = &d0;
	if (USERMODE(ar0[PS]))
		type |= USER;
	u.u_traptype = 0;
	switch (type) {

	/*
	 * Trap not expected.
	 * Usually a kernel mode bus error.
	 */
	default:
crash:
		panicstr = "trap";	/* fake it for printfs */
		snap();
		printf("ps_kind 0x%x\n", ps_kind.voffset);
		panic(NULL);


	case 0x108>>2:
	case (0x108>>2)+USER:
		printf("Kilroy was here\n");
		pccabort();
		break;
	case SYSCALL + USER:	/* sys call */
	{
		register *a;
		register struct sysent *callp;

		u.u_error = 0;
		ar0[PS] &= ~PS_C;
		a = (int *)ar0[SP];	/* point to first arg */
		a++;		/* skip word with param count */
		i = ar0[R0] & 0377;
		if (i >= NSYSENT)
			i = 0;
		else if (i == 0) {	/* indirect */
			i = fuword(a++)&0377;
			if (i >= NSYSENT)
				i = 0;
		}
		callp = &sysent[i];
		for (i = 0; i < callp->sy_narg; i++)
			u.u_arg[i] = fuword(a++);
		u.u_dirp = (caddr_t)u.u_arg[0];
		u.u_rval1 = 0;
		u.u_rval2 = ar0[R1];
		u.u_ap = u.u_arg;
		if (setjmp(u.u_qsav)) {
			if (u.u_stflag)
				stclrlck();
			if (u.u_error == 0)
				u.u_error = EINTR;
		} else
			(*callp->sy_call)();
		if (u.u_error) {
			ar0[R0] = u.u_error;
			ar0[PS] |= PS_C;	/* carry bit */
		} else {
			ar0[R0] = u.u_rval1;
			ar0[R1] = u.u_rval2;
		}
	}
	/*
	 * If the system call trap was executed with the T bit set,
	 * the trace trap will be ignored.  So, the bit is reset
	 * and the SIGTRAP signal is sent to the process.
	 */
	if (ar0[PS] & PS_T) {	/* was trace bit set ? */
		i = SIGTRAP;
		ar0[PS] &= ~PS_T;	/* reset trace bit */
		break;		/* send signal to process */
	}
	goto out;


	case RESCHED + USER:	/* Allow process switch */
		qswtch();
		goto out;

	/*
	 * This condition will happen when a TRAP instruction is 
	 * executed with the T bit set.  This will only be taken
	 * care of by the system if it happens while doing a system
	 * call.  All other occurences of this phenomenon will cause
	 * the breakppoint to be ignored.
	 */
	case TRCTRAP:		/* trace trap in kernel mode */
		return;		/* ignore this trap */

	case FPETRAP + USER:	/* floating point exception */
		i = SIGFPE;
		break;

	case ZDVDERR + USER:	/* zero divide check */
		i = SIGFPE;
		break;

	case CHKTRAP + USER:	/* CHK and CHK2 (M68020 only) trap */
		i = SIGFPE;
		break;

	case ADDRERR + USER:	/* address error */
		i = SIGBUS;
		u.u_traptype = TRAPADDR;
		break;

	case TRAPVFT + USER:	/* TRAPV instruction */
		i = SIGFPE;
		break;

	case IOTTRAP + USER:	/* IOT trap */
		i = SIGIOT;
		break;

	case EMTTRAP + USER:	/* EMT trap */
		i = SIGEMT;
		break;

	case FMTERR + USER:	/* bad exception format */
		i = SIGILL;
		break;
	case UNINTR:		/* uninitialized interrupt;fall thru to STRAY */
	case UNINTR + USER:	/* ####### NOTE ##### may move */


	case STRAYFT:		/* spurious interrupt */
	case STRAYFT + USER:
		stray(type<<2);
		return;
	case COPROTV + USER:	/* coprocessor protocol violation trap */
		i = SIGILL;
		break;
	/*
	 * If the user SP is below the stack segment,
	 * grow the stack automatically.
	 */
	case BUSERR + USER:	/* user bus error */
		if (grow(ar0[SP]))
			goto out;
		u.u_traptype = TRAPBUS;
		i = SIGSEGV;
		break;

	case BPTFLT + USER:	/* bpt instruction fault */
		ar0[PC] -= 2;	/* back up pc for simulated breakpoint */
	case TRCTRAP + USER:	/* trace trap */
		ar0[PS] &= ~PS_T;	/* turn off trace bit */
		i = SIGTRAP;
		break;

	case PRIVFLT + USER:	/* privileged instruction fault */
	case L1111FT + USER:	/* Line 1111 emulator trap */
	case L1010FT + USER:	/* Line 1010 emulator trap */
	case INSTERR + USER:	/* Illegal instruction */
		i = SIGILL;
		break;
	}
	psignal(u.u_procp, i);

out:
	if (u.u_procp->p_sig && issig())
		psig();
	curpri = setpri(u.u_procp);
	if (runrun) {
		qswtch();
	}
	if (u.u_prof.pr_scale)
		addupc((caddr_t)u.u_ar0[PC], &u.u_prof, (int)(u.u_stime-syst));
}

/*
 * nonexistent system call-- signal bad system call.
 */
nosys()
{
	u.u_error = EINVAL;
}

/*
 * Ignored system call
 */
nullsys()
{
}

stray(addr)
{
	printf("stray interrupt at %x\n", addr);
}
snap()		/* print registsers */
{
	register long *ar;

	ar = u.u_ar0;
	printf("user = ");
	printf("0x%x\n", u.u_procp->p_addr);
	printf("ps = 0x%x\n", (ar[PS]>>16) & 0xffff);
	printf("pc = 0x%x\n", ar[PC]);
	printf("d0: %x\n", ar[R0]);
	printf("d1: %x\n", ar[R1]);
	printf("d2: %x\n", ar[R2]);
	printf("d3: %x\n", ar[R3]);
	printf("d4: %x\n", ar[R4]);
	printf("d5: %x\n", ar[R5]);
	printf("d6: %x\n", ar[R6]);
	printf("d7: %x\n", ar[R7]);
	printf("a0: %x\n", ar[R8]);
	printf("a1: %x\n", ar[R9]);
	printf("a2: %x\n", ar[R10]);
	printf("a3: %x\n", ar[R11]);
	printf("a4: %x\n", ar[R12]);
	printf("a5: %x\n", ar[R13]);
	printf("a6: %x\n", ar[R14]);
	printf("usp: %x\n", ar[SP]);
}

get_spl()		/* return current priority level */
{
	asm("	movw	sr,d0");
}
